package ru.yandex.autotests.directapi.configuration;

import java.lang.management.ManagementFactory;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import one.util.streamex.StreamEx;
import org.eclipse.jetty.jmx.MBeanContainer;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.NetworkConnector;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.handler.HandlerCollection;
import org.eclipse.jetty.server.handler.StatisticsHandler;
import org.eclipse.jetty.util.thread.QueuedThreadPool;
import org.eclipse.jetty.webapp.WebAppContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.jetty.JettyConfig;
import ru.yandex.direct.common.jetty.JettyLauncher;
import ru.yandex.direct.utils.HostPort;
import ru.yandex.misc.ip.IpPortUtils;

public class TestJettyLauncher extends JettyLauncher {
    private static final int MAX_REQUEST_SIZE = 16 * 1024 * 1024;
    private static final Logger logger = LoggerFactory.getLogger(JettyLauncher.class);
    private final Lock startupLock;
    private final Condition startedCondition;
    private HostPort hostPort;

    private TestJettyLauncher(JettyConfig conf) {
        super(conf);
        startupLock = new ReentrantLock();
        startedCondition = startupLock.newCondition();
    }

    public static TestJettyLauncher server(JettyConfig conf) {
        return new TestJettyLauncher(conf);
    }

    @Override
    public void run() throws Exception {
        Server server = configureServer();
        try {
            startupLock.lock();

            HandlerCollection handlers = new HandlerCollection();
            for (WebAppContext ctx : webAppContexts) {
                // если не можем запустить приложение - не запускаем и jetty
                ctx.setThrowUnavailableOnStartupException(true);
                ctx.setInitParameter("org.eclipse.jetty.servlet.Default.dirAllowed", "false");
                ctx.setMaxFormContentSize(MAX_REQUEST_SIZE);
                handlers.addHandler(ctx);
            }
            if (conf.requestLogEnabled) {
                handlers.addHandler(createRequestLogHandler());
            }

            // Needed for graceful shutdown, see Server.setStopTimeout
            StatisticsHandler statsHandler = new StatisticsHandler();
            statsHandler.setHandler(handlers);
            server.setHandler(statsHandler);

            try {
                server.start();
                hostPort = StreamEx.of(server.getConnectors())
                        .select(NetworkConnector.class)
                        .map(c -> new HostPort(Optional.ofNullable(c.getHost()).orElse("localhost"), c.getLocalPort()))
                        .findFirst()
                        .orElseThrow(() -> new RuntimeException("Unable to start Jetty"));
                logger.info("jetty started, ports={}", hostPort.getPort());
            } catch (Exception e) {
                server.stop();
                throw new JettyLaunchException("Can't start Jetty", e);
            } finally {
                startedCondition.signalAll();
            }
        } finally {
            startupLock.unlock();
        }

        try {
            server.join();
            logger.info("jetty stopped");
        } finally {
            server.stop();
        }
    }

    Lock getStartupLock() {
        return startupLock;
    }

    HostPort hostPort() throws InterruptedException {
        if (hostPort == null) {
            try {
                startupLock.lock();
                startedCondition.await(90, TimeUnit.SECONDS);
                if (hostPort == null) {
                    throw new JettyLaunchException("Can't get Jetty host and port");
                }
            } finally {
                startupLock.unlock();
            }
        }
        return hostPort;
    }

    private Server configureServer() {
        QueuedThreadPool threadPool = configureThreadPool();
        Server server = new Server(threadPool);

        // выставляем статистики наружу через JMX
        MBeanContainer mbContainer = new MBeanContainer(ManagementFactory.getPlatformMBeanServer());
        server.addEventListener(mbContainer);
        server.addBean(mbContainer);

        ServerConnector connector =
                new ServerConnector(server, new HttpConnectionFactory(getHttpConfiguration()));
        connector.setIdleTimeout(conf.idleTimeout);
        connector.setPort(IpPortUtils.getFreeLocalPort().getPort());

        server.addConnector(connector);

        setTermHandler();

        return server;
    }

    List<WebAppContext> getWebAppContexts() {
        return webAppContexts;
    }
}
