package ru.yandex.autotests.directapi.model.api5.general;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;

import com.yandex.direct.api.v5.general.ActionResult;
import org.apache.commons.beanutils.BeanMap;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;

import ru.yandex.autotests.direct.utils.matchers.BeanCompareStrategy;
import ru.yandex.autotests.direct.utils.matchers.BeanEquals;
import ru.yandex.direct.common.TranslationService;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class ExpectedResult extends BeanMap {
    public static final String ID = "id";
    public static final String WARNINGS = "warnings";
    public static final String ERRORS = "errors";

    @Nonnull
    private final Matcher<? super Long> idMatcher;
    @Nonnull
    private final Collection<TranslatableNotification> expectedErrors;
    @Nonnull
    private final Collection<TranslatableNotification> expectedWarnings;
    @Nonnull
    private BeanCompareStrategy resultCompareStrategy = new BeanCompareStrategy();

    private ExpectedResult(@Nonnull Matcher<? super Long> idMatcher,
                           @Nonnull Collection<TranslatableNotification> expectedErrors,
                           @Nonnull Collection<TranslatableNotification> expectedWarnings) {
        super();
        setBean(new ActionResult());
        this.idMatcher = idMatcher;
        this.expectedErrors = expectedErrors;
        this.expectedWarnings = expectedWarnings;

        this.put(WARNINGS, expectedWarnings);
        this.put(ERRORS, expectedErrors);
    }

    public static ExpectedResult success() {
        return new ExpectedResult(notNullValue(), emptyList(), emptyList());
    }

    public static ExpectedResult success(long id) {
        return new ExpectedResult(is(id), emptyList(), emptyList());
    }

    public static ExpectedResult errors(Notification... errors) {
        var translatedErrors = mapList(Arrays.asList(errors),
                TranslatableNotification::fromNotification);

        return new ExpectedResult(nullValue(), translatedErrors, emptyList());
    }

    public static ExpectedResult errorsIgnoringDetails(Notification... errors) {
        var translatableErrors = Arrays.stream(errors)
                .map(n -> TranslatableNotification.fromNotification(n).ignoringDetails())
                .collect(toList());
        return new ExpectedResult(nullValue(), translatableErrors, emptyList());
    }

    public static ExpectedResult errorsIgnoringDetailsAndMessage(Notification... errors) {
        var translatableErrors = Arrays.stream(errors)
                .map(n -> TranslatableNotification.fromNotification(n).ignoringDetails().ignoringMessage())
                .collect(toList());
        return new ExpectedResult(nullValue(), translatableErrors, emptyList());
    }

    public static ExpectedResult errorWithCode(int expectedErrorCode) {
        var error = new Notification(expectedErrorCode, null);
        return errorsIgnoringDetailsAndMessage(error);
    }

    public static ExpectedResult warningsIgnoringDetailsAndMessage(Notification... warnings) {
        var translatableWarnings = Arrays.stream(warnings)
                .map(n -> TranslatableNotification.fromNotification(n).ignoringDetails().ignoringMessage())
                .collect(toList());
        return new ExpectedResult(notNullValue(), emptyList(), translatableWarnings);
    }

    public Matcher<ActionResult> getResultMatcher(TranslationService translationService) {
        List<Matcher<? super ActionResult>> matchers = new ArrayList<>();

        matchers.add(hasProperty(ID, idMatcher));

        if (expectedErrors.isEmpty()) {
            matchers.add(hasProperty(ERRORS, emptyIterable()));
        } else {
            matchers.add(hasProperty(ERRORS, contains(expectedErrors.stream()
                    .map(n -> n.getMatcher(translationService))
                    .collect(toList()))));
        }

        if (expectedWarnings.isEmpty()) {
            matchers.add(hasProperty(WARNINGS, emptyIterable()));
        } else {
            matchers.add(hasProperty(WARNINGS, contains(expectedWarnings.stream()
                    .map(n -> n.getMatcher(translationService))
                    .collect(toList()))));
        }

        return Matchers.allOf(matchers);
    }

    public BeanCompareStrategy getActionResultCompareStrategy() {
        return resultCompareStrategy;
    }

    public ExpectedResult withResultCompareStrategy(BeanCompareStrategy resultCompareStrategy) {
        this.resultCompareStrategy = resultCompareStrategy;
        return this;
    }

    public Matcher<ActionResult> getActionResultMatcher() {
        return BeanEquals.beanEquals((ActionResult) this.getBean())
                .accordingStrategy(this.getActionResultCompareStrategy());
    }

}
