package ru.yandex.autotests.directapi.model.api5.general;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.yandex.direct.api.v5.general.ActionResult;
import com.yandex.direct.api.v5.general.MultiIdsActionResult;
import org.apache.commons.beanutils.BeanMap;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;

import ru.yandex.direct.common.TranslationService;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;

public class MultiIdsExpectedResult extends BeanMap {
    private static final String IDS = "ids";
    private static final String WARNINGS = "warnings";
    private static final String ERRORS = "errors";

    private final boolean expectSuccess;

    @Nullable
    private final List<Long> ids;
    @Nonnull
    private final List<TranslatableNotification> expectedErrors;
    @Nonnull
    private final List<TranslatableNotification> expectedWarnings;

    private MultiIdsExpectedResult(boolean expectSuccess,
                                   @Nullable List<Long> ids,
                                   @Nonnull List<TranslatableNotification> expectedErrors,
                                   @Nonnull List<TranslatableNotification> expectedWarnings) {
        super();
        setBean(new ActionResult());
        this.ids = ids;
        this.expectSuccess = expectSuccess;
        this.expectedErrors = expectedErrors;
        this.expectedWarnings = expectedWarnings;

        this.put(WARNINGS, expectedWarnings);
        this.put(ERRORS, expectedErrors);
    }

    public static MultiIdsExpectedResult success() {
        return new MultiIdsExpectedResult(true, null, emptyList(), emptyList());
    }

    public static MultiIdsExpectedResult success(List<Long> ids) {
        return new MultiIdsExpectedResult(true, ids, emptyList(), emptyList());
    }

    public static MultiIdsExpectedResult errors(Notification... notifications) {
        List<TranslatableNotification> warnings = new ArrayList<>();
        List<TranslatableNotification> errors = new ArrayList<>();

        for (Notification notification : notifications) {
            errors.add(TranslatableNotification.fromNotification(notification));
        }

        return new MultiIdsExpectedResult(false, null, errors, warnings);
    }

    public Matcher<MultiIdsActionResult> getResultMatcher(TranslationService translationService) {
        List<Matcher<? super MultiIdsActionResult>> matchers = new ArrayList<>();

        if (expectSuccess) {
            matchers.add(hasProperty(IDS, ids == null ? notNullValue() : is(equalTo(ids))));
        } else {
            matchers.add(hasProperty(IDS, emptyIterable()));
        }

        if (expectedErrors == null) {
            matchers.add(hasProperty(ERRORS, nullValue()));
        } else if (expectedErrors.isEmpty()) {
            matchers.add(hasProperty(ERRORS, emptyIterable()));
        } else {
            matchers.add(hasProperty(ERRORS, contains(expectedErrors.stream()
                    .map(n -> n.getMatcher(translationService))
                    .collect(toList()))));
        }

        if (expectedWarnings == null) {
            matchers.add(hasProperty(WARNINGS, nullValue()));
        } else if (expectedWarnings.isEmpty()) {
            matchers.add(hasProperty(WARNINGS, emptyIterable()));
        } else {
            matchers.add(hasProperty(WARNINGS, contains(expectedWarnings.stream()
                    .map(n -> n.getMatcher(translationService))
                    .collect(toList()))));
        }

        return Matchers.allOf(matchers);
    }
}
