package ru.yandex.autotests.directapi.steps;

import java.util.Set;

import org.junit.rules.TestWatcher;

import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.authentication.DummyAuthenticationContainer;
import ru.yandex.autotests.directapi.model.campaigns.MetrikaGoals;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.core.testing.steps.Steps;
import ru.yandex.direct.core.testing.stub.MetrikaClientStub;
import ru.yandex.direct.feature.FeatureName;

public class ApiSteps extends TestWatcher {

    public final UserSteps userSteps;
    private final ApiStepsState state;
    private final Steps coreSteps;
    private final MetrikaClientStub metrikaClientStub;
    private ConnectionConfig connectionConfig;

    public ApiSteps() {
        this.state = new ApiStepsState();
        this.connectionConfig = new ConnectionConfig("http://" + this.state.getApi().hostPort.toString());
        this.userSteps = userSteps();

        var context = state.getApplicationContext();
        this.coreSteps = context.getBean(Steps.class);
        this.metrikaClientStub = context.getBean(MetrikaClientStub.class);

        prepareDb();
    }

    // TODO: выполнять prepareDb один раз при запуске тестов, эти настройки глобальные
    private void prepareDb() {
        // в репозитории onDuplicateIgnore
        coreSteps.trustedRedirectSteps().addValidCounters();

        metrikaClientStub.addCounterGoal(MetrikaGoals.VALID_COUNTER_ID, (int) MetrikaGoals.VALID_GOAL_ID);
        metrikaClientStub.addCounterGoal(MetrikaGoals.INACCESSIBLE_COUNTER_ID, (int) MetrikaGoals.INACCESSIBLE_GOAL_ID);
        metrikaClientStub.addUnavailableCounter(MetrikaGoals.INACCESSIBLE_COUNTER_ID);

        var ppcPropertiesSteps = userSteps.ppcPropertiesSteps();
        ppcPropertiesSteps.set(PpcPropertyNames.CROSS_DEVICE_DEFAULT_ATTRIBUTION_TYPE_ENABLED, true);
        ppcPropertiesSteps.set(PpcPropertyNames.MEANINGFUL_GOAL_VALUE_SERIALIZE_AS_STRING, true);
    }

    public ClientInfo createClient() {
        ClientInfo client = coreSteps.clientSteps().createDefaultClient();
        DummyAuthenticationContainer.createTokenForLogin(client.getLogin());
        state.setClientInfo(client);
        return client;
    }

    private void prepareClient(ClientInfo client) {
        var featureSteps = userSteps.featureSteps();
        featureSteps.setClientFeature(client, FeatureName.CROSS_DEVICE_ATTRIBUTION_TYPES, true);
        // без этой фичи все цели считаются автоматически доступными
        featureSteps.setClientFeature(client, FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED, true);

        metrikaClientStub.addUserCounter(client.getUid(), MetrikaGoals.VALID_COUNTER_ID);
        metrikaClientStub.addGoals(client.getUid(), Set.of(MetrikaGoals.VALID_GOAL));
    }

    public ApiSteps withConnection(ConnectionConfig connectionConfig) {
        this.connectionConfig = connectionConfig;
        return this;
    }

    public ApiSteps as(ClientInfo client) {
        DummyAuthenticationContainer.createTokenForLogin(client.getLogin());
        state.setClientInfo(client);
        prepareClient(client);
        return this;
    }

    public ApiSteps as(User user) {
        DummyAuthenticationContainer.createTokenForLogin(user.getLogin());
        state.setUser(user);
        return this;
    }

    public ApiSteps asUnknownClient() {
        var client = createClient();
        return as(client);
    }

    public UserSteps userSteps() {
        if (connectionConfig == null) {
            throw new RuntimeException("Connection not set");
        }
        return new UserSteps(state, connectionConfig);
    }

    public Steps coreSteps() {
        return coreSteps;
    }
}
