package ru.yandex.autotests.directapi.steps;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import com.yandex.direct.api.v5.general.ActionResult;
import com.yandex.direct.api.v5.general.MultiIdsActionResult;
import org.apache.commons.lang3.StringUtils;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.springframework.context.i18n.LocaleContextHolder;

import ru.yandex.autotests.direct.utils.textresource.ITextResource;
import ru.yandex.autotests.directapi.apiclient.ApiClient;
import ru.yandex.autotests.directapi.apiclient.JsonClient;
import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorMatcher;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorMessage;
import ru.yandex.autotests.directapi.apiclient.errors.Api5JsonError;
import ru.yandex.autotests.directapi.apiclient.errors.Api5JsonErrorMatcher;
import ru.yandex.autotests.directapi.apiclient.internal.MethodInvocationResult;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.ServiceNames;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.MultiIdsExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.direct.common.TranslationService;
import ru.yandex.qatools.allure.annotations.Step;

import static java.util.Arrays.asList;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.either;
import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.fail;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

public class BaseApiSteps {
    protected final ConnectionConfig connectionConfig;
    protected final RequestHeader requestHeader;
    protected final ApiStepsState state;
    private final TranslationService translationService;

    public BaseApiSteps(ConnectionConfig connectionConfig,
                        RequestHeader requestHeader, ApiStepsState state) {
        this.connectionConfig = connectionConfig;
        this.requestHeader = requestHeader;
        this.state = state;
        this.translationService = state.getApplicationContext().getBean(TranslationService.class);
    }

    public ApiClient defaultClientV5() {
        ProtocolType protocolType = connectionConfig.getProtocolType();
        switch (protocolType) {
            case JSON:
                return jsonClientV5();
            case SOAP:
                return soapClientV5();
        }
        throw new DirectAPIException("Установлен неизвестный тип клиента для выполнения запросов - " + protocolType);
    }

    public JsonClient jsonClientV5() {
        return new JsonClient(connectionConfig, requestHeader);
    }

    public ApiClient soapClientV5() {
        throw new DirectAPIException("Установлен неизвестный тип клиента для выполнения запросов - SOAP");
    }

    public Matcher<Api5JsonError> getApi5JsonErrorMatcher(Notification error) {
        return getApi5JsonErrorMatcher(error.getErrorCode(), error.getDetailsResource(), error.getParams());
    }

    public Matcher<Api5JsonError> getApi5JsonErrorMatcher(int errorCode,
                                                          ITextResource expectedDetails,
                                                          Object... params) {
        return getApi5JsonErrorMatcher(errorCode, expectedDetails, List.of(params));
    }

    private Matcher<Api5JsonError> getApi5JsonErrorMatcher(int errorCode,
                                                           ITextResource expectedDetails,
                                                           List<Object> params) {
        var expectedMessage = Api5ErrorMessage.fromNotificationCode(errorCode);
        LocaleContextHolder.setLocale(Locale.forLanguageTag(requestHeader.getLocale()));
        var expectedError = new Api5JsonError()
                .setErrorCode(errorCode)
                .setMessage(translationService.translate(expectedMessage.getTranslatable()))
                .setDetails(translationService.translate(expectedDetails.getTranslatable(params)));
        return equalTo(expectedError);
    }

    protected <T, ResultType> ArrayList<ResultType> extractActionResults(T response, Action action) {
        assumeThat("получен ответ сервиса", response, notNullValue());
        String structureName = StringUtils.capitalize(action.toString()) + "Results";
        String methodToCall = "get" + structureName;
        java.lang.reflect.Method resultsGetter;
        try {
            resultsGetter = response.getClass().getMethod(methodToCall);
        } catch (NoSuchMethodException noMethodEx) {
            throw new DirectAPIException("Метод API5 вернул ответ " + response.getClass().getName() + ", " +
                    "в котором отсутствует структура " + structureName, noMethodEx);
        }
        ArrayList<ResultType> results;
        try {
            results = (ArrayList<ResultType>) resultsGetter.invoke(response);
        } catch (IllegalAccessException accessExc) {
            throw new DirectAPIException(
                    "Ошибка доступа при вызове метода " + methodToCall + "в классе " + response.getClass(), accessExc);
        } catch (InvocationTargetException invTargetExc) {
            throw new DirectAPIException("Ошибка reflection при вызове метода " + methodToCall, invTargetExc);
        }
        return results;
    }

    protected void checkActionResults(List<? extends ActionResult> actualResults, ExpectedResult... expectedResults) {
        Matcher<Iterable<? extends ActionResult>> resultMatcher =
                getActionResultListMatcher(asList(expectedResults));
        assertThat("содержание результатов соответствует ожидаемому", actualResults, resultMatcher);
    }

    private Matcher<Iterable<? extends ActionResult>> getActionResultListMatcher(
            List<ExpectedResult> expectedResults) {
        List<Matcher<? super ActionResult>> expectedMatchers = new ArrayList<>();
        LocaleContextHolder.setLocale(Locale.forLanguageTag(requestHeader.getLocale()));

        for (ExpectedResult expResult : expectedResults) {
            expectedMatchers.add(expResult.getResultMatcher(translationService));
        }

        return contains(expectedMatchers);
    }

    protected void checkMultiIdsActionResults(List<MultiIdsActionResult> actualResults,
                                              MultiIdsExpectedResult... expectedResults) {
        Matcher<Iterable<? extends MultiIdsActionResult>> resultMatcher =
                getMultiIdsActionResultListMatcher(asList(expectedResults));
        assertThat("содержание результатов соответствует ожидаемому", actualResults, resultMatcher);
    }

    private Matcher<Iterable<? extends MultiIdsActionResult>> getMultiIdsActionResultListMatcher(
            List<MultiIdsExpectedResult> expectedResults) {
        List<Matcher<? super MultiIdsActionResult>> expectedMatchers = new ArrayList<>();
        LocaleContextHolder.setLocale(Locale.forLanguageTag(requestHeader.getLocale()));

        for (MultiIdsExpectedResult expResult : expectedResults) {
            expectedMatchers.add(expResult.getResultMatcher(translationService));
        }
        return contains(expectedMatchers);
    }

    public Object shouldGetResultOn(
            ServiceNames serviceName, String login, Action action, Object args, ExpectedResult... expectedResults) {
        MethodInvocationResult<?> response =
                defaultClientV5().invokeMethodEx(serviceName, login, action, args);
        ArrayList<ActionResult> actualResult = extractActionResults(response.getResponseObject(), action);
        checkActionResults(actualResult, expectedResults);
        return response.getResponseObject();
    }


    @Step(value = "Проверим, что метод возвращает ошибку с заданным кодом. {0}")
    public void shouldGetErrorWithCodeOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, Object args,
            Integer expectedErrorCode) {
        try {
            defaultClientV5().invokeMethod(serviceName, login, action, args);
            assertThat("получена ошибка", null, CoreMatchers.equalTo(expectedErrorCode));
        } catch (Api5Error | Api5JsonError error) {
            Integer actualErrorCode = error.getErrorCode();
            assertThat("совпал код", actualErrorCode, CoreMatchers.equalTo(expectedErrorCode));
        }
    }

    @Step(value = "Проверим, что метод возвращает ошибку, не сравнивая details. {0}")
    public void shouldGetErrorOnIgnoringDetails(
            String assertMessage, ServiceNames serviceName, String login, Action action, Object args,
            Api5Error expectedError) {
        shouldGetErrorOn(assertMessage, serviceName, login, action, args, expectedError, true);
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, Object args,
            Api5Error expectedError) {
        shouldGetErrorOn(assertMessage, serviceName, login, action, args, expectedError, false);
    }

    private void shouldGetErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, Object args,
            Api5Error expectedError,
            boolean ignoreDetails) {
        try {
            defaultClientV5().invokeMethod(serviceName, login, action, args);
            assertThat("получена ошибка", null, CoreMatchers.equalTo(expectedError));
        } catch (Api5Error soapError) {
            if (expectedError == null) {
                fail("Ожидался успех, а была получена ошибка " + soapError);
            } else {
                if (ignoreDetails) {
                    assertThat("совпал текст ошибки (SOAP)",
                            soapError,
                            Api5ErrorMatcher.equalToIgnoreLocaleAndDetails(expectedError, List.of()));
                } else {
                    assertThat("совпал текст ошибки (SOAP)",
                            soapError,
                            Api5ErrorMatcher.equalToIgnoreLocale(expectedError, List.of()));
                }
            }
        } catch (Api5JsonError jsonError) {
            if (expectedError == null) {
                fail("Ожидался успех, а была получена ошибка " + jsonError);
            } else {
                if (ignoreDetails) {
                    assertThat("совпал code и message ошибки (JSON)",
                            jsonError,
                            Api5JsonErrorMatcher.equalToIgnoreLocaleAndDetails(expectedError.toJsonError(),
                                    List.of()));
                } else {
                    assertThat("совпал текст ошибки (JSON)",
                            jsonError,
                            Api5JsonErrorMatcher.equalToIgnoreLocale(expectedError.toJsonError(),
                                    List.of()));
                }
            }
        }
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, Object args,
            Api5Error expectedError, Api5Error alternativeError) {
        try {
            defaultClientV5().invokeMethod(serviceName, login, action, args);
            assertThat("получена ошибка", null, CoreMatchers.equalTo(expectedError));
        } catch (Api5Error soapError) {
            if (expectedError == null) {
                fail("Ожидался успех, а была получена ошибка " + soapError);
            } else {
                assertThat("совпал текст ошибки (SOAP)",
                        soapError,
                        either(Api5ErrorMatcher
                                .equalToIgnoreLocale(expectedError, List.of()))
                                .or(Api5ErrorMatcher.equalToIgnoreLocale(alternativeError,
                                        List.of())));
            }
        } catch (Api5JsonError jsonError) {
            if (expectedError == null) {
                fail("Ожидался успех, а была получена ошибка " + jsonError);
            } else {
                assertThat("совпал текст ошибки (JSON)",
                        jsonError,
                        either(Api5JsonErrorMatcher.equalToIgnoreLocale(expectedError.toJsonError(),
                                List.of()))
                                .or(Api5JsonErrorMatcher.equalToIgnoreLocale(alternativeError.toJsonError(),
                                        List.of())));
            }
        }
    }

    public void shouldGetErrorOn(
            ServiceNames serviceName, String login, Action action, Object args, Api5Error expectedError) {
        shouldGetErrorOn("", serviceName, login, action, args, expectedError);
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetJSONErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, Object args,
            Api5Error expectedError) {
        try {
            if (args != null) {
                defaultClientV5().invokeMethod(serviceName, login, action, args);
            } else {
                jsonClientV5().invokeMethod(serviceName, login, action, null);
            }
            assertThat("получена ошибка", null, CoreMatchers.equalTo(expectedError));
        } catch (Api5JsonError jsonError) {
            assertThat("совпал текст ошибки (JSON)",
                    jsonError,
                    Api5JsonErrorMatcher.equalToIgnoreLocale(
                            expectedError.toJsonError(),
                            List.of()));
        }
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetJSONErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, Object args,
            Api5Error expectedError, Api5Error alternativeError) {
        try {
            jsonClientV5().invokeMethod(serviceName, login, action, args);
            assertThat("получена ошибка", null,
                    either(CoreMatchers.equalTo(expectedError)).or(CoreMatchers.equalTo(alternativeError)));
        } catch (Api5JsonError jsonError) {
            assertThat("совпал текст ошибки (JSON)",
                    jsonError,
                    either(Api5JsonErrorMatcher.equalToIgnoreLocale(expectedError.toJsonError(),
                            List.of()))
                            .or(Api5JsonErrorMatcher.equalToIgnoreLocale(alternativeError.toJsonError(),
                                    List.of())));
        }
    }

    public void shouldGetJSONErrorOn(
            ServiceNames serviceName, String login, Action action, Object args, Api5Error expectedError) {
        shouldGetJSONErrorOn("", serviceName, login, action, args, expectedError);
    }

    public void shouldGetJSONErrorOn(
            ServiceNames serviceName, String login, Action action, Object args, Api5Error expectedError,
            Api5Error alternativeError) {
        shouldGetJSONErrorOn("", serviceName, login, action, args, expectedError, alternativeError);
    }
}
