package ru.yandex.autotests.directapi.steps;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.direct.api.v5.campaigns.AddResponse;
import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignFundsEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.campaigns.CampaignStateEnum;
import com.yandex.direct.api.v5.campaigns.CampaignsSelectionCriteria;
import com.yandex.direct.api.v5.campaigns.GetResponse;
import com.yandex.direct.api.v5.general.ActionResult;
import one.util.streamex.StreamEx;
import org.apache.commons.beanutils.BeanMap;
import org.hamcrest.Matchers;

import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.ServiceNames;
import ru.yandex.autotests.directapi.model.api5.campaigns.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.campaigns.MetrikaGoals;
import ru.yandex.autotests.irt.testutils.allure.AssumptionException;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.qatools.allure.annotations.Step;

import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.directapi.model.api5.Action.ADD;
import static ru.yandex.autotests.directapi.model.api5.ServiceNames.CAMPAIGNS;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

public class CampaignSteps extends BaseApiSteps {
    private static final ObjectMapper MAPPER = new ObjectMapper();

    public CampaignSteps(ApiStepsState state, ConnectionConfig connectionConfig) {
        super(connectionConfig, new RequestHeader(state.getClientLogin(), state.getToken()), state);
    }

    private AddResponse campaignsAdd(AddRequestMap parameters, String login) {
        return defaultClientV5().invokeMethod(CAMPAIGNS, login, ADD, parameters.getBean());
    }

    private AddResponse campaignsAdd(Object parameters, String login) {
        return defaultClientV5().invokeMethod(CAMPAIGNS, login, ADD, parameters);
    }

    public AddResponse campaignsAdd(AddRequestMap parameters) {
        return campaignsAdd(parameters, null);
    }

    public List<Long> campaignsAddWithCheck(String login, AddRequestMap request) {
        List<ActionResult> results = campaignsAdd(request, login).getAddResults();
        lookForResponseExceptions(results);
        return getIDsFromActionResults(results);
    }

    public List<Long> campaignsAddWithCheck(AddRequestMap request) {
        return campaignsAddWithCheck(null, request);
    }

    public List<Long> campaignsAddWithCheck(String login, CampaignAddItemMap... campaigns) {
        AddRequestMap request = new AddRequestMap().withCampaigns(campaigns);
        return campaignsAddWithCheck(login, request);
    }

    public List<Long> campaignsAddWithCheck(CampaignAddItemMap... campaigns) {
        return campaignsAddWithCheck(null, campaigns);
    }

    public AddResponse shouldGetResultOnAdd(
            AddRequestMap parameters, String login, ExpectedResult... expectedResults) {
        return (AddResponse) shouldGetResultOn(login, Action.ADD, parameters.getBean(), expectedResults);
    }

    public AddResponse shouldGetResultOnAdd(AddRequestMap parameters, ExpectedResult... expectedResults) {
        return shouldGetResultOnAdd(parameters, null, expectedResults);
    }

    public AddResponse shouldGetResultOnAdd(Object parameters, ExpectedResult... expectedResults) {
        return shouldGetResultOnAdd(parameters, null, expectedResults);
    }

    public AddResponse shouldGetResultOnAdd(Object parameters, String login, ExpectedResult[] expectedResults) {
        return (AddResponse) shouldGetResultOn(login, Action.ADD, parameters, expectedResults);
    }

    public void expectErrorOnCampaignsAdd(AddRequestMap parameters, String login, Api5Error api5Error) {
        shouldGetErrorOn(ServiceNames.CAMPAIGNS, login, Action.ADD, parameters, api5Error);
    }

    public void expectErrorOnCampaignsAddIgnoringDetails(AddRequestMap parameters, String login, Api5Error api5Error) {
        shouldGetErrorOnIgnoringDetails("", ServiceNames.CAMPAIGNS, login, Action.ADD, parameters,
                api5Error);
    }

    public void expectErrorOnCampaignsAdd(AddRequestMap parameters, Api5Error api5Error) {
        expectErrorOnCampaignsAdd(parameters, null, api5Error);
    }

    public void expectErrorOnCampaignsAddIgnoringDetails(AddRequestMap parameters, Api5Error api5Error) {
        shouldGetErrorOnIgnoringDetails("", ServiceNames.CAMPAIGNS, null, Action.ADD, parameters,
                api5Error);
    }

    public void expectErrorOnCampaignsAdd(Object parameters, String login, Api5Error api5Error) {
        shouldGetJSONErrorOn(ServiceNames.CAMPAIGNS, login, Action.ADD, parameters, api5Error);
    }

    public void expectErrorOnCampaignsAdd(Object parameters, Api5Error api5Error) {
        expectErrorOnCampaignsAdd(parameters, null, api5Error);
    }


    public Long addCampaign(CampaignAddItemMap campaignAddItemMap, String login) {
        var results = campaignsAdd(new AddRequestMap().withCampaigns(campaignAddItemMap), login)
                .getAddResults();
        var ids = results.stream()
                .map(ActionResult::getId)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        assumeThat("создана одна кампания", ids, Matchers.hasSize(1));
        return ids.get(0);
    }

    public Long addCampaign(CampaignAddItemMap campaignAddItemMap) {
        return addCampaign(campaignAddItemMap, null);
    }

    public Long addCampaign(Object params, String login) {
        AddResponse response = campaignsAdd(params, login);
        List<Long> ids = extractIdsFromAddResponse(response);
        assumeThat("создана одна кампания", ids, hasSize(1));
        return ids.get(0);
    }

    public Long addCampaign(Object params) {
        return addCampaign(params, null);
    }

    private List<Long> extractIdsFromAddResponse(AddResponse addResponse) {
        return StreamEx.of(addResponse.getAddResults())
                .map(ActionResult::getId)
                .toList();
    }

    public Long addDefaultTextCampaign() {
        return addDefaultTextCampaign(state.getClientInfo());
    }

    public Long addDefaultTextCampaign(ClientInfo clientInfo) {
        return addDefaultTextCampaign(clientInfo.getLogin());
    }

    public Long addDefaultTextCampaign(String login) {
        var addItem = new CampaignAddItemMap().defaultCampaignAddItem().withDefaultTextCampaign();
        return addCampaign(addItem, login);
    }

    public Long addDefaultMobileAppCampaign() {
        return addDefaultMobileAppCampaign(state.getClientInfo());
    }

    public Long addDefaultMobileAppCampaign(ClientInfo clientInfo) {
        var addItem = new CampaignAddItemMap().defaultCampaignAddItem().withDefaultMobileAppCampaign();
        return addCampaign(addItem, clientInfo.getLogin());
    }

    public Long addDefaultDynamicTextCampaign() {
        return addDefaultDynamicTextCampaign(state.getClientInfo());
    }

    public Long addDefaultDynamicTextCampaign(ClientInfo clientInfo) {
        var addItem = new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withDefaultDynamicTextCampaign();
        return addCampaign(addItem, clientInfo.getLogin());
    }

    public Long addDefaultCampaign(CampaignType type) {
        return addDefaultCampaign(state.getClientInfo(), type);
    }

    public Long addDefaultSmartCampaign(ClientInfo clientInfo) {
        var addItem = new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withDefaultSmartCampaign(MetrikaGoals.VALID_COUNTER_ID);
        return addCampaign(addItem, clientInfo.getLogin());
    }

    public Long addDefaultCampaign(ClientInfo clientInfo, CampaignType type) {
        switch (type) {
            case TEXT:
                return addDefaultTextCampaign(clientInfo);
            case MOBILE_CONTENT:
                return addDefaultMobileAppCampaign(clientInfo);
            case DYNAMIC:
                return addDefaultDynamicTextCampaign(clientInfo);
            default:
                throw new AssertionError("Unimplemented");
        }
    }

    public <T extends BeanMap> Object shouldGetResultOn(Action action,
                                                        T args,
                                                        ExpectedResult... expectedResults) {
        return shouldGetResultOn(null, action, args, expectedResults);
    }

    //region Get
    @Step("[Campaigns]: Get")
    public GetResponse campaignsGet(GetRequestMap parameters, String login) {
        return defaultClientV5()
                .invokeMethod(ServiceNames.CAMPAIGNS, login, Action.GET, parameters.getBean());
    }

    public GetResponse campaignsGet(GetRequestMap parameters) {
        return campaignsGet(parameters, null);
    }

    public List<CampaignGetItem> getCampaigns(String login, GetRequestMap request) {
        return campaignsGet(request, login).getCampaigns();
    }

    public List<CampaignGetItem> getCampaigns(GetRequestMap request) {
        return getCampaigns(null, request);
    }

    @Step
    public List<CampaignGetItem> getCampaigns(String login, Long... campaignIds) {
        GetResponse getResponse = campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteria()
                                        .withIds(campaignIds))
                        .withAllFieldNames()
                        .withAllTextCampaignFieldNames(),
                login);
        return getResponse.getCampaigns();
    }

    @Step
    public List<CampaignGetItem> getCampaigns(Long... campaignIds) {
        return getCampaigns(null, campaignIds);
    }

    /**
     * Получить информацию о кампании по id. Запрашиваются все поля для текстовых кампаний.
     */
    public CampaignGetItem getCampaign(long campaignId) {
        return getCampaign(null, campaignId);
    }

    /**
     * Получить информацию о кампании по id. Запрашиваются все поля для текстовых кампаний.
     *
     * @param login Client-Login запроса
     */
    public CampaignGetItem getCampaign(String login, long campaignId) {
        List<CampaignGetItem> campaigns = getCampaigns(login, campaignId);
        assumeThat("Получена одна кампания", campaigns, hasSize(1));
        return campaigns.get(0);
    }

    /**
     * Получить кол-во средств на кампании.
     * <p>
     * Выкидывает {@link AssumptionException} в случае, если подключен общий счёт
     *
     * @param login      Client-Login запроса
     * @param campaignId идентификатор кампании, кол-во средств на которой нужно получить
     * @return кол-во средств в валюте кампании
     */
    public Money getCampaignSum(String login, long campaignId) {
        var getItem = campaignsGet(login, campaignId, CampaignFieldEnum.FUNDS, CampaignFieldEnum.CURRENCY);
        assumeThat("Кампания не должна быть подключена к ОС",
                getItem.getFunds().getMode(), not(equalTo(CampaignFundsEnum.SHARED_ACCOUNT_FUNDS)));
        var currency = CurrencyCode.valueOf(getItem.getCurrency().value());
        return Money.valueOfMicros(getItem.getFunds().getCampaignFunds().getSum(), currency);
    }

    public List<Long> getAllCampaignIds(String login) {
        GetResponse getResponse = campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteria())
                        .withFieldNames(CampaignFieldEnum.ID),
                login);
        return getResponse.getCampaigns().stream().map(CampaignGetItem::getId).collect(Collectors.toList());
    }

    public void verifyHasNonArchivedCampaign(String login) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withStates(CampaignStateEnum.ON, CampaignStateEnum.OFF, CampaignStateEnum.ENDED,
                        CampaignStateEnum.SUSPENDED);
        GetResponse getResponse = campaignsGet(
                new GetRequestMap().withSelectionCriteria(selectionCriteria).withFieldNames(CampaignFieldEnum.ID),
                login);
        boolean hasCampaigns = getResponse.getCampaigns().stream().map(CampaignGetItem::getId).findAny().isPresent();
        if (!hasCampaigns) {
            addDefaultTextCampaign(login);
        }
    }

    public List<Long> getAllCampaignIds() {
        return getAllCampaignIds(null);
    }

    public List<Long> getNotConvertedCampaignIds(String login) {
        GetResponse getResponse = campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteria()
                                        .withStates(Stream.of(CampaignStateEnum.values())
                                                .filter(s -> !s.equals(CampaignStateEnum.CONVERTED))
                                                .toArray(CampaignStateEnum[]::new)))
                        .withFieldNames(CampaignFieldEnum.ID),
                login);
        return getResponse.getCampaigns().stream().map(CampaignGetItem::getId).collect(Collectors.toList());
    }

    public List<Long> getNotConvertedCampaignIds() {
        return getNotConvertedCampaignIds(null);
    }

    @Step("[Campaigns]: Get: получить поля кампании по ID")
    public CampaignGetItem campaignsGet(String login, Long id, CampaignFieldEnum... fieldNames) {
        GetResponse getResponse = campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteria()
                                        .withIds(id)
                        )
                        .withFieldNames(fieldNames),
                login);
        assumeThat("вернулась 1 кампания", getResponse.getCampaigns(), hasSize(1));
        return getResponse.getCampaigns().get(0);
    }

    public CampaignGetItem campaignsGet(Long id, CampaignFieldEnum... fieldNames) {
        return campaignsGet(null, id, fieldNames);
    }


    public void expectErrorOnCampaignsGet(GetRequestMap parameters, String login, Api5Error api5Error) {
        shouldGetErrorOn(ServiceNames.CAMPAIGNS, login, Action.GET, parameters, api5Error);
    }

    public void expectErrorOnCampaignsGet(GetRequestMap parameters, Api5Error api5Error) {
        expectErrorOnCampaignsGet(parameters, null, api5Error);
    }

    public void expectErrorOnCampaignsGet(Object parameters, String login, Api5Error api5Error) {
        shouldGetJSONErrorOn(ServiceNames.CAMPAIGNS, login, Action.GET, parameters, api5Error);
    }

    public void expectErrorOnCampaignsGet(Object parameters, Api5Error api5Error) {
        shouldGetJSONErrorOn(ServiceNames.CAMPAIGNS, null, Action.GET, parameters, api5Error);
    }
    //endregion

    public <T extends BeanMap> Object shouldGetResultOn(String login, Action action, T args,
                                                        ExpectedResult... expectedResults) {
        return shouldGetResultOn(login, action, args == null ? null : args.getBean(), expectedResults);
    }

    @Step("Проверяем ID/Warnings/Errors в ответе сервиса Campaigns")
    public Object shouldGetResultOn(String login, Action action, Object args,
                                    ExpectedResult... expectedResults) {
        Object response = defaultClientV5()
                .invokeMethod(ServiceNames.CAMPAIGNS, login, action, args);
        ArrayList<ActionResult> actualResult = extractActionResults(response, action);
        if (action.equals(Action.DELETE)) {
            List<Long> ids = getIDsFromActionResults(actualResult);
        }
        if (action.equals(Action.ADD)) {
            List<Long> ids = getIDsFromActionResults(actualResult);
        }
        checkActionResults(actualResult, expectedResults);
        return response;
    }

    private List<Long> getIDsFromActionResults(List<ActionResult> results) {
        return StreamEx.of(results)
                .map(ActionResult::getId)
                .toList();
    }

    private void lookForResponseExceptions(List<ActionResult> results) {
        var errors = StreamEx.of(results)
                .filter(x -> !x.getErrors().isEmpty())
                .toList();

        assertThat("отсутствуют ошибки при вызове метода", errors, emptyIterable());
    }
}
