package ru.yandex.direct.domain.retargeting

import ru.yandex.direct.domain.client.ClientID

/**
 * Defines identifier type for [RetargetingCondition].
 */
@JvmInline
value class RetargetingConditionID(val value: Long) : Comparable<RetargetingConditionID> {
    override fun compareTo(other: RetargetingConditionID) = value.compareTo(other.value)
}

/**
 * Свойства условия ретаргетинга (некоторые свойства дублируют информацию, уже содержащуюся в условии ретаргетина или его типе).
 */
enum class RetargetingConditionProperty {
    /**
     * Условие содержит только вычитание целей.
     *
     * @see <a href="https://a.yandex-team.ru/arcadia/commit/a91d00a0a5471395342799cfb7f1bea5f6f770da#file-trunk/db_schema/ppc/retargeting_conditions.text">Коммит a91d00a0a5471395342799cfb7f1bea5f6f770da</a>
     * @see <a href="https://st.yandex-team.ru/DIRECT-51427">DIRECT-51427: Корректировка "Не был на сайте"</a>
     */
    NEGATIVE,

    /**
     * Маркирует условия ретаргетинга на интересы.
     *
     * @see <a href="https://a.yandex-team.ru/arcadia/commit/fb585445fda150b13afba39ce8a08ffbc65796c3#file-trunk/db_schema/ppc/retargeting_conditions.schema.sql">Коммит fb585445fda150b13afba39ce8a08ffbc65796c3</a>
     */
    INTEREST,

    /**
     * Маркирует ретаргетинги добавленные автоматически (в отличие от остальных добавленных вручную).
     *
     * @see <a href="https://a.yandex-team.ru/arcadia/commit/r8033670">Коммит r8033670</a>
     */
    AUTORETARGETING
}

/**
 * Общий sealed-класс условий ретаргетингов.
 */
sealed class RetargetingCondition(
    val id: RetargetingConditionID,

    val clientId: ClientID,

    private val properties: Set<RetargetingConditionProperty> = setOf(),
)

// TODO: Уточнить ограничения для всех типов ниже. Дополнить описание классов.

/**
 * Ретаргетинг по интересам. Может иметь только одно условие с одной целью на интерес.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-56850">DIRECT-56850: РМП: Таргетирование на интересы к приложениям</a> (2016 год)
 * @see <a href="https://wiki.yandex-team.ru/direct/technicaldesign/target-interests/">Таргетирование на интересы пользователя</a>
 */
class InterestsRetargetingCondition(
    id: RetargetingConditionID, clientId: ClientID,
    // TODO: В базе есть записи с несколькими Rule для retargeting_conditions с типом interests
    val rule: InterestsRule
) : RetargetingCondition(id, clientId)

/**
 * Ретаргетинг по целям метрики, самый первый тип ретаргетинга.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-15172">DIRECT-15172: Ретаргетинг</a>
 */
class MetrikaRetargetingCondition(
    id: RetargetingConditionID, clientId: ClientID,
    val rules: Collection<MetrikaRule>,
    val name: String = "",
    val description: String = ""
) : RetargetingCondition(id, clientId)

/**
 * Ретаргетинг по данным ДМП
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-70401">DIRECT-70401: Условия показа по сегментам крипты</a>
 * @see <a href="https://wiki.yandex-team.ru/direct/projects/crypta/#tipyuslovijjpodboraauditorijjdirekta">Типы условий подбора аудиторий директа (от 2017 года)</a>
 */
class DMPRetargetingCondition(id: RetargetingConditionID, clientId: ClientID, val rules: Collection<GenericRule>) :
    RetargetingCondition(id, clientId)

/**
 * Ретаргетинг по АБ-сегментам
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-76428">DIRECT-76428: Использование сегментов для экспериментов в Директе</a>
 */
class ABSegmentsRetargetingCondition(
    id: RetargetingConditionID,
    clientId: ClientID,
    val rules: Collection<GenericRule>
) : RetargetingCondition(id, clientId)

/**
 * Ретаргетинг по Brand Safety категориям.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-108715">DIRECT-108715: Создание условия для retargeting_conditions с BrandSafety категориями</a>
 * @see <a href="https://st.yandex-team.ru/DIRECT-107987'>DIRECT-107987: Brand Safety в Директе [Запрещенный конент на кампанию]</a>
 */
class BrandSafetyRetargetingCondition(
    id: RetargetingConditionID,
    clientId: ClientID,
    val rules: Collection<GenericRule>
) : RetargetingCondition(id, clientId)

/**
 * Ретаргетинг по геосегментам.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-117872">DIRECT-117872: Добавить новый тип условия ретаргетинга для гиперлокальности</a>
 * @see <a href="https://st.yandex-team.ru/DIRECT-115821">DIRECT-115821: Гиперлокальный директ</a>
 */
class GeoSegmentsRetargetingCondition(
    id: RetargetingConditionID,
    clientId: ClientID,
    val rules: Collection<GenericRule>
) : RetargetingCondition(id, clientId)

/**
 * Ретаргетинг по шорткатам.
 * @see <a href="https://st.yandex-team.ru/DIRECT-149398">DIRECT-149398: Шорткаты условий ретаргетинга</a>
 * @see <a href="https://st.yandex-team.ru/DIRECT-148179">DIRECT-148179: Переделка ретаргетинга</a>
 */
class ShortcutsRetargetingCondition(
    id: RetargetingConditionID,
    clientId: ClientID,
    val rules: Collection<GenericRule>
) : RetargetingCondition(id, clientId)
