package ru.yandex.direct.domain.retargeting

import kotlin.reflect.full.companionObjectInstance
import kotlin.time.Duration
import kotlin.time.Duration.Companion.days

/**
 * Интерфейс фабрики GoalID создающий конкретный класс GoalID по Long.
 */
interface GoalIDFactory<T : GoalID> {
    /**
     * Получение конкретного типа идентификатора из [Long].
     */
    fun from(id: Long): GoalID?

    /**
     * Получение случайного идентификатора (используется в тестах).
     */
    fun random(): T
}

/**
 * Идентификаторы целей.
 *
 * Диапазон разбит на отрезки, которые выдаются разным источникам целей, но есть и старые назначенные вручную цели (см. [MobileAppSpecialGoalID]).
 * Все диапазоны были описаны [на вики](https://wiki.yandex-team.ru/JurijjGalickijj/raznoe/goalid/).
 */
sealed class GoalID(val value: Long) {
    companion object : GoalIDFactory<GoalID> {
        private val TYPES by lazy {
            GoalID::class.sealedSubclasses.mapNotNull { it.companionObjectInstance }
                .filterIsInstance<GoalIDFactory<GoalID>>()
        }

        override fun from(id: Long) = TYPES.firstNotNullOfOrNull { it.from(id) }

        override fun random() = TYPES.random().random()
    }

    override fun equals(other: Any?): Boolean =
        when (other) {
            is GoalID -> value == other.value
            else -> false
        }

    override fun hashCode() = value.hashCode()
}

/**
 * Реализация GoalIDFactory для конструирования GoalID по валидному отрезку идентификаторов.
 */
open class GoalIDRangeFactory<T : GoalID>(val idRange: LongRange, val constructor: (Long) -> T) : GoalIDFactory<T> {
    override fun from(id: Long) = if (id in idRange) constructor(id) else null

    override fun random() = constructor(idRange.random())
}

/**
 * Sealed-класс для всех целей с доступными для них полями.
 */
sealed class Goal {
    /**
     * Количество дней (от 1 до 540), за которое проверяется выполнение цели.
     * Для сегментов Яндекс Метрики и сегментов Яндекс Аудиторий параметр не используется, переданное значение игнорируется.
     */
    abstract val membershipLifeSpan: Duration
}

/**
 * Маркировочный sealed-класс для целей из метрики.
 */
sealed class GoalFromMetrika : Goal()

/**
 * Маркировочный sealed-класс для целей из директа.
 */
sealed class GoalFromDirect : Goal()

/**
 * Маркировочный sealed-класс для целей из крипты.
 */
sealed class GoalsFromCrypta : Goal() {
    final override val membershipLifeSpan: Duration
        get() = 0.days
}

/**
 * Заданные вручную цели в директе для мобильных приложений (РМП).
 *
 * @see <a href="https://a.yandex-team.ru/arcadia/direct/perl/protected/Stat/Const.pm?rev=r9708405#L445-472">Константы в перловом коде</a>
 * @see <a href="https://st.yandex-team.ru/DIRECT-82131">DIRECT-82131: РМП: Ассоциированные инстолы: Для CPI стратегии позволить выбирать ассоциированные инсталлы (внутренние роли)</a>
 * @see <a href="https://st.yandex-team.ru/DIRECT-77554">DIRECT-77554: Показывать ассоциированные инстолы в интерфейсе директа</a>
 */
data class MobileAppSpecialGoal(
    val id: ID,
    override val membershipLifeSpan: Duration = 1.days
) : GoalFromMetrika() {
    /**
     * ID для [MobileAppSpecialGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        enum class Type(val value: Long) {
            UNKNOWN(-1),
            APP_INSTALL_ANTIFRAUD(3),
            APP_INSTALL(4),
            APP_INSTALL_TRACKER(5),
            ASSISTED_APP_INSTALL(6),
            ASSISTED_APP_INSTALL_WITHOUT_REINSTALLATION(7),
            ACHIEVED(38402972),
            APP_LAUNCHED(38403008),
            ADDED_PAYMENT_INFO(38403053),
            ADDED_TO_CART(38403071),
            ADDED_TO_WISHLIST(38403080),
            COMPLETED_REGISTRATION(38403095),
            COMPLETED_TUTORIAL(38403104),
            INITIATED_CHECKOUT(38403131),
            PURCHASED(38403173),
            RATED(38403191),
            SEARCHED(38403197),
            SPENT_CREDITS(38403206),
            UNLOCKED_ACHIEVEMENT(38403215),
            VIEWED_CONTENT(38403230),
            SPENT_TIME_IN_APP(38403338),
            SHARED(38403494),
            EVENT_1(38403530),
            EVENT_2(38403545),
            EVENT_3(38403581)
        }

        val type: Type
            get() = ID_MAP[value] ?: Type.UNKNOWN

        companion object : GoalIDFactory<ID> {
            private val ID_MAP = Type.values().associateBy(Type::value)
            private val ID_SET = Type.values().map(Type::value).toSet()

            override fun from(id: Long) = if (id in ID_SET) ID(id) else null

            override fun random() = ID(ID_SET.random())
        }
    }
}

/**
 * Цели из Метрики.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-15172">DIRECT-15172: Ретаргетинг</a>
 */
data class MetrikaGoal(
    val id: ID,
    override val membershipLifeSpan: Duration = 1.days
) : GoalFromMetrika() {
    /**
     * ID целей для [MetrikaGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(0L..999_999_999L, ::ID) {
            override fun from(id: Long): ID? {
                // Так как диапазон MetrikaGoal.ID перекрывает MobileAppGoalID, то не возвращаем их как MobileGoal.ID
                MobileAppSpecialGoal.ID.from(id)?.let { return null }

                return if (id in idRange) ID(id) else null
            }

            override fun random(): ID {
                return generateSequence { idRange.random() }.firstNotNullOf { from(it) }
            }
        }
    }
}

/**
 * Сегменты в Метрике.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-37874">DIRECT-37874: Расширенный ретаргетинг по сегментам</a>
 */
data class SegmentGoal(val id: ID) : GoalFromMetrika() {
    /**
     * ID целей для [SegmentGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(1_000_000_000L..1_499_999_999L, ::ID)
    }

    override val membershipLifeSpan: Duration
        get() = 540.days
}

/**
 * Look-a-like цели в директе.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-106789">DIRECT-106789: ☂️ Расширение сегментов LAL-ом</a>
 */
data class LalSegmentGoal(
    val id: ID, override val membershipLifeSpan: Duration = 1.days
) : GoalFromMetrika() {
    /**
     * ID для [LalSegmentGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(1_500_000_000L..1_899_999_999L, ::ID)
    }
}

/**
 * "Мобильные цели" в директе.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-135973">DIRECT-135973: ☂ Оптимизация и ретаргетинг на цели из трекеров</a>
 */
data class MobileGoal(
    val id: ID, override val membershipLifeSpan: Duration = 1.days
) : GoalFromMetrika() {
    /**
     * ID для [MobileGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(1_900_000_000L..1_999_999_999L, ::ID)
    }
}

/**
 * Аудитории в Я.Аудитории.
 */
data class AudienceGoal(val id: ID) : GoalFromMetrika() {
    /**
     * ID для [AudienceGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(2_000_000_000L..2_498_999_999L, ::ID)
    }

    override val membershipLifeSpan: Duration
        get() = 540.days
}

/**
 * СоцДем из крипты.
 *
 * @see <a href="https://wiki.yandex-team.ru/direct/projects/crypta/?from=%2Fusers%2Faliho%2Fprojects%2Fdirect%2Fcrypta%2F#direkt">Использование сегментов крипты в директе</a>
 */
data class SocialDemoGoal(val id: ID) : GoalsFromCrypta() {
    /**
     * ID для [SocialDemoGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(2_499_000_000L..2_499_000_099L, ::ID)
    }
}

/**
 * Цели для расширенного СоцДема (Семья/Профессия/...).
 *
 * @see <a href="https://wiki.yandex-team.ru/direct/projects/crypta/?from=%2Fusers%2Faliho%2Fprojects%2Fdirect%2Fcrypta%2F#diapazondljarasshirennyjjsocdemsemja/professija/">Диапазон для Расширенного соцдема</a>
 */
data class FamilyGoal(val id: ID) : GoalsFromCrypta() {
    /**
     * ID для [FamilyGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(2_499_000_100L..2_499_000_199L, ::ID)
    }
}

/**
 * Цели для поведенческих признаков (Пользуются смартфонами на iOS/Мотоциклисты/...).
 *
 * @see <a href="https://wiki.yandex-team.ru/direct/projects/crypta/?from=%2Fusers%2Faliho%2Fprojects%2Fdirect%2Fcrypta%2F#diapazondljapovedencheskixpriznakov">Диапазон для поведенческих признаков</a>
 */
data class BehaviorsGoal(val id: ID) : GoalsFromCrypta() {
    /**
     * ID для [BehaviorsGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(2_499_000_200L..2_499_001_099L, ::ID)
    }
}

/**
 * Цели для интересов (Спорт/Плавание/...).
 *
 * @see <a href="https://wiki.yandex-team.ru/direct/projects/crypta/?from=%2Fusers%2Faliho%2Fprojects%2Fdirect%2Fcrypta%2F#diapazondljainteresov">Диапазон для интересов</a>
 */
data class InterestsGoal(val id: ID) : GoalsFromCrypta() {
    /**
     * ID для [InterestsGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(2_499_001_100L..2_499_979_999L, ::ID)
    }
}

/**
 * Цели соответствующие внутренним сегментам Крипты.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-114425">DIRECT-114425: Добавить новые сегменты в cryptaSegments для внутренней рекламы</a>
 * @see <a href="https://st.yandex-team.ru/DIRECT-112633">DIRECT-112633: Адаптация cryptaSegments к требованиям внутренней рекламы</a>
 */
data class InternalGoal(val id: ID) : GoalsFromCrypta() {
    /**
     * ID для [InternalGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(2_499_980_000L..2_499_989_999L, ::ID)
    }
}

/**
 * Цели, соответстующие музыкальным жанрам.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-96776">DIRECT-96776: [audio][java-web] Добавить фильтрацию по новому типу сегментов (аудио жанры)</a>
 * @see <a href="https://st.yandex-team.ru/DIRECT-94995">DIRECT-94995: 🎧 Аудиореклама в Директе. Технический запуск. 10 Июня 2019.</a>
 */
data class AudioGenresGoal(val id: ID) : GoalsFromCrypta() {
    /**
     * ID для [AudioGenresGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(2_499_990_000L..2_499_999_999L, ::ID)
    }
}

/**
 * Цели для экспериментальных сегментов Аудиторий. На базе аудиторий сделано AB тестирование.
 *
 * @see <a href="https://st.yandex-team.ru/AUDIENCE-578">AUDIENCE-578: Экспериментальные сегменты в Метрике (API)</a>
 */
data class ABSegmentGoal(
    val id: ID, override val membershipLifeSpan: Duration = 1.days
) : GoalFromMetrika() {
    /**
     * ID для [ABSegmentGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(2_500_000_000L..2_599_999_999L, ::ID)
    }
}

/**
 * Цели для сегментов CDP (Customer Data Platform).
 *
 * @see <a href="https://st.yandex-team.ru/CDP-308">CDP-308: Ретаргетинг по данным CDP</a>
 */
data class CDPSegmentGoal(val id: ID) : GoalFromMetrika() {
    /**
     * ID для [CDPSegmentGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(2_600_000_000L..2_999_999_999L, ::ID)
    }

    override val membershipLifeSpan: Duration
        get() = 540.days
}

/**
 * Цели ecommerce-покупки счетчика в Метрике.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-69393">DIRECT-69393: eCommerce цели: поддержать eCommerce цели в ретаргетингах</a>
 */
data class ECommereceGoal(
    val id: ID, override val membershipLifeSpan: Duration = 1.days
) : GoalFromMetrika() {
    /**
     * ID для [ECommereceGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(3_000_000_000L..3_899_999_999L, ::ID)
    }
}

/**
 * Цели для категории Brand Safety (запрещенный контент на заказ Директа и одно из условий показа в Директе).
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-76632">DIRECT-76632: Brand safety</a>
 */
data class BrandSafetyGoal(val id: ID) : GoalsFromCrypta() {
    /**
     * ID для [BrandSafetyGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(4_294_967_296L..4_294_967_296L + 999L, ::ID)
    }
}

/**
 * Цели для категории контента.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-105581">DIRECT-105581: Таргетинг на категории и жанры</a>
 */
data class ContentCategoryGoal(val id: ID) : GoalsFromCrypta() {
    /**
     * ID для [ContentCategoryGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(4_294_967_296L + 1_000L..4_294_967_296L + 2_999L, ::ID)
    }
}

/**
 * ID для жанра контента.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-105581">DIRECT-105581: Таргетинг на категории и жанры</a>
 */
data class ContentGenreGoal(val id: ID) : GoalsFromCrypta() {
    /**
     * ID для [ContentGenreGoal]
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(4_294_967_296L + 3_000L..4_294_967_296L + 4_999L, ::ID)
    }
}

/**
 * ID для целей привязанных к хостам из крипты ([таблица в YT](https://yt.yandex-team.ru/hahn/navigation?path=//home/crypta/production/siberia/custom_audience/suggester/hosts))
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-158257"></a>
 * @see <a href="https://st.yandex-team.ru/DIRECT-158254">DIRECT-158254: Custom Audience в Директе Vol. 2</a>
 */
data class HostGoal(
    val id: ID, override val membershipLifeSpan: Duration = 1.days
) : GoalFromDirect() {
    /**
     * ID для [HostGoal].
     */
    class ID private constructor(val id: Long) : GoalID(id) {
        companion object : GoalIDRangeFactory<ID>(19_000_000_000L..19_899_999_999L, ::ID)
    }
}
