package ru.yandex.direct.domain.retargeting

import ru.yandex.direct.domain.adgroup.AdGroupID
import ru.yandex.direct.domain.campaign.CampaignID
import ru.yandex.direct.libs.nonEmptyCollection.NonEmptyCollection

/**
 * Условия выбора ретаргетингов. Список трактуется как объединение условий через AND.
 */
class SelectionCriteria(c: NonEmptyCollection<Criteria>) : NonEmptyCollection<Criteria> by c {
    /**
     * Применяет все критерии к списку ретаргетингов и оставляет только те, которые подошли под все условия одновременно.
     */
    fun apply(retargetings: Collection<Retargeting>) =
        map { criteria ->
            when (criteria) {
                is IDsCriteria -> retargetings.filter { it.id in criteria.ids }
                is AdGroupIDsCriteria -> retargetings.filter { it.adGroupID in criteria.ids }
                is CampaignIDsCriteria -> retargetings.filter { it.campaignID in criteria.ids }
                is RetargetingConditionIDsCriteria -> retargetings.filter { it.condition.id in criteria.ids }
                is SuspendedCriteria -> retargetings.filter { it.suspended == criteria.suspended }
                is InterestIDsCriteria -> retargetings.filter {
                    (it.condition is InterestsRetargetingCondition) &&
                        (it.condition.rule.goal.id in criteria.ids)
                }
            }.toSet()
        }.reduce { acc, it -> acc intersect it }
}

sealed class Criteria

/**
 * Критерий фильтрует ретаргетинги по их [RetargetingID].
 */
class IDsCriteria(val ids: NonEmptyCollection<RetargetingID>) : Criteria() {}

/**
 * Критерий фильтрует ретаргетинги по их [AdGroupID].
 */
class AdGroupIDsCriteria(val ids: NonEmptyCollection<AdGroupID>) : Criteria() {}

/**
 * Критерий фильтрует ретаргетинги по их [CampaignID].
 */
class CampaignIDsCriteria(val ids: NonEmptyCollection<CampaignID>) : Criteria() {}

/**
 * Критерий фильтрует ретаргетинги по их [RetargetingConditionID].
 */
class RetargetingConditionIDsCriteria(val ids: NonEmptyCollection<RetargetingConditionID>) : Criteria() {}

/**
 * Критерий фильтрует ретаргетинги по интересам.
 */
class InterestIDsCriteria(val ids: NonEmptyCollection<InterestsGoal.ID>) : Criteria() {}

/**
 * Критерий фильтрует ретаргетинги по их статусу [suspended].
 */
class SuspendedCriteria(val suspended: Boolean) : Criteria() {}
