package ru.yandex.direct.infrastructure.mysql.column_types

import org.jetbrains.exposed.sql.Column
import org.jetbrains.exposed.sql.CustomFunction
import org.jetbrains.exposed.sql.EqOp
import org.jetbrains.exposed.sql.ExpressionWithColumnType
import org.jetbrains.exposed.sql.GreaterOp
import org.jetbrains.exposed.sql.LongColumnType
import org.jetbrains.exposed.sql.StringColumnType
import org.jetbrains.exposed.sql.Table
import org.jetbrains.exposed.sql.intLiteral
import org.jetbrains.exposed.sql.stringParam
import kotlin.reflect.full.isSubclassOf

inline fun <reified T : Enum<T>> safeValueOf(type: String): T? {
    return try {
        java.lang.Enum.valueOf(T::class.java, type)
    } catch (e: IllegalArgumentException) {
        null
    }
}

/**
 * Creates a set column with custom SQL type.
 * The main usage is to use a database specific type.
 *
 * See [Table.customEnumeration] and [https://github.com/JetBrains/Exposed/wiki/DataTypes#how-to-use-database-enum-types] for more details.
 *
 * @param name The column name
 * @param sql A SQL definition for the column
 * @param fromDb A lambda to convert a value received from a database to a set instance
 * @param toDb A lambda to convert a set instance to a value which will be stored to a database
 */
@Suppress("UNCHECKED_CAST")
fun <T : Enum<T>> Table.customSet(
    name: String,
    sql: String,
    fromDb: (Any) -> Set<T>,
    toDb: (Set<T>) -> Any
): Column<Set<T>> =
    registerColumn(name,
        object : StringColumnType() {
            override fun sqlType(): String = sql
            override fun valueFromDB(value: Any): Any =
                if (value::class.isSubclassOf(Set::class)) value else fromDb(value)

            override fun notNullValueToDB(value: Any): Any = toDb(value as Set<T>)
            override fun nonNullValueToString(value: Any): String = super.nonNullValueToString(notNullValueToDB(value))
        }
    )

infix fun <T : Enum<T>> ExpressionWithColumnType<Set<T>>.contains(enum: T) =
    GreaterOp(
        CustomFunction<LongColumnType>("FIND_IN_SET", this.columnType, stringParam(enum.name.lowercase()), this),
        intLiteral(0)
    )

infix fun <T : Enum<T>> ExpressionWithColumnType<Set<T>>.notContains(enum: T) =
    EqOp(
        CustomFunction<LongColumnType>("FIND_IN_SET", this.columnType, stringParam(enum.name.lowercase()), this),
        intLiteral(0)
    )
