package ru.yandex.direct.infrastructure.mysql.column_types

import org.jetbrains.exposed.sql.Column
import org.jetbrains.exposed.sql.ColumnType
import org.jetbrains.exposed.sql.LongColumnType
import org.jetbrains.exposed.sql.Table
import org.jetbrains.exposed.sql.TextColumnType
import kotlin.reflect.KClass

@Suppress("UNCHECKED_CAST")
abstract class WrapperColumnType<Raw : Any, Wrapper : Any>(
    val rawColumnType: ColumnType,
    val rawClazz: KClass<Raw>,
    val wrapperClazz: KClass<Wrapper>,
    val instanceCreator: (Raw) -> Wrapper,
    val valueExtractor: (Wrapper) -> Raw,
) : ColumnType() {

    override fun sqlType(): String = rawColumnType.sqlType()

    override fun valueToString(value: Any?): String = when (value) {
        null -> {
            check(nullable) { "NULL in non-nullable column" }
            "NULL"
        }
        else -> nonNullValueToString(value)
    }

    override fun valueFromDB(value: Any): Wrapper = rawColumnType.valueFromDB(value).let { rawValue ->
        when {
            rawClazz.isInstance(rawValue) -> instanceCreator(rawValue as Raw)
            else -> error("Raw value from database $rawValue of class ${rawValue::class.qualifiedName} is not valid $rawClazz")
        }
    }

    override fun notNullValueToDB(value: Any) = when {
        wrapperClazz.isInstance(value) -> rawColumnType.notNullValueToDB(valueExtractor(value as Wrapper))
        else -> error("Wrapper value $value of class ${value::class.qualifiedName} is not valid $wrapperClazz")
    }

    override fun nonNullValueToString(value: Any) = when {
        wrapperClazz.isInstance(value) -> rawColumnType.nonNullValueToString(valueExtractor(value as Wrapper))
        else -> error("Raw value $value of class ${value::class.qualifiedName} is not valid $wrapperClazz")
    }

    override fun toString() =
        "WrapperColumnType(rawColumnType=$rawColumnType, rawClazz=$rawClazz, wrapperClazz=$wrapperClazz)"
}

class LongWrapperColumnType<Wrapper : Any>(
    wrapperClazz: KClass<Wrapper>,
    instanceCreator: (Long) -> Wrapper,
    valueExtractor: (Wrapper) -> Long,
) : WrapperColumnType<Long, Wrapper>(LongColumnType(), Long::class, wrapperClazz, instanceCreator, valueExtractor)

class StringWrapperColumnType<Wrapper : Any>(
    wrapperClazz: KClass<Wrapper>,
    instanceCreator: (String) -> Wrapper,
    valueExtractor: (Wrapper) -> String,
) : WrapperColumnType<String, Wrapper>(TextColumnType(), String::class, wrapperClazz, instanceCreator, valueExtractor)

inline fun <reified Wrapper : Any> Table.longWrapper(
    name: String,
    noinline instanceCreator: (Long) -> Wrapper,
    noinline valueExtractor: (Wrapper) -> Long,
): Column<Wrapper> = registerColumn(name, LongWrapperColumnType(Wrapper::class, instanceCreator, valueExtractor))

inline fun <reified Wrapper : Any> Table.stringWrapper(
    name: String,
    noinline instanceCreator: (String) -> Wrapper,
    noinline valueExtractor: (Wrapper) -> String,
): Column<Wrapper> = registerColumn(name, StringWrapperColumnType(Wrapper::class, instanceCreator, valueExtractor))
