package ru.yandex.direct.infrastructure.mysql.repositories

import org.jetbrains.exposed.sql.ResultRow
import org.jetbrains.exposed.sql.StdOutSqlLogger
import org.jetbrains.exposed.sql.addLogger
import org.jetbrains.exposed.sql.and
import org.jetbrains.exposed.sql.insert
import org.jetbrains.exposed.sql.not
import org.jetbrains.exposed.sql.select
import org.jetbrains.exposed.sql.transactions.transaction
import ru.yandex.direct.domain.client.ClientID
import ru.yandex.direct.domain.retargeting.ABSegmentGoal
import ru.yandex.direct.domain.retargeting.ABSegmentsRetargetingCondition
import ru.yandex.direct.domain.retargeting.AudienceGoal
import ru.yandex.direct.domain.retargeting.AudioGenresGoal
import ru.yandex.direct.domain.retargeting.BehaviorsGoal
import ru.yandex.direct.domain.retargeting.BrandSafetyGoal
import ru.yandex.direct.domain.retargeting.BrandSafetyRetargetingCondition
import ru.yandex.direct.domain.retargeting.CDPSegmentGoal
import ru.yandex.direct.domain.retargeting.ContentCategoryGoal
import ru.yandex.direct.domain.retargeting.ContentGenreGoal
import ru.yandex.direct.domain.retargeting.DMPRetargetingCondition
import ru.yandex.direct.domain.retargeting.ECommereceGoal
import ru.yandex.direct.domain.retargeting.FamilyGoal
import ru.yandex.direct.domain.retargeting.GenericRule
import ru.yandex.direct.domain.retargeting.GeoSegmentsRetargetingCondition
import ru.yandex.direct.domain.retargeting.Goal
import ru.yandex.direct.domain.retargeting.GoalID
import ru.yandex.direct.domain.retargeting.HostGoal
import ru.yandex.direct.domain.retargeting.InterestsGoal
import ru.yandex.direct.domain.retargeting.InterestsRetargetingCondition
import ru.yandex.direct.domain.retargeting.InterestsRule
import ru.yandex.direct.domain.retargeting.InternalGoal
import ru.yandex.direct.domain.retargeting.LalSegmentGoal
import ru.yandex.direct.domain.retargeting.MetrikaGoal
import ru.yandex.direct.domain.retargeting.MetrikaRetargetingCondition
import ru.yandex.direct.domain.retargeting.MetrikaRule
import ru.yandex.direct.domain.retargeting.MobileAppSpecialGoal
import ru.yandex.direct.domain.retargeting.MobileGoal
import ru.yandex.direct.domain.retargeting.RetargetingCondition
import ru.yandex.direct.domain.retargeting.RetargetingConditionID
import ru.yandex.direct.domain.retargeting.RetargetingConditionProperty
import ru.yandex.direct.domain.retargeting.RetargetingConditionRepository
import ru.yandex.direct.domain.retargeting.SegmentGoal
import ru.yandex.direct.domain.retargeting.ShortcutsRetargetingCondition
import ru.yandex.direct.domain.retargeting.SocialDemoGoal
import ru.yandex.direct.infrastructure.mysql.column_types.contains
import ru.yandex.direct.infrastructure.mysql.column_types.notContains
import ru.yandex.direct.infrastructure.mysql.sharding.ShardedDatabase
import ru.yandex.direct.infrastructure.mysql.tables.ConditionJSON
import ru.yandex.direct.infrastructure.mysql.tables.ConditionJSONRule
import ru.yandex.direct.infrastructure.mysql.tables.ConditionType
import ru.yandex.direct.infrastructure.mysql.tables.RetargetingConditions
import kotlin.time.Duration.Companion.days

private fun ConditionJSONRule.toGoals(): List<Goal> =
    this.goals.mapNotNull { (goal_id, time) ->
        when (val goalID = GoalID.from(goal_id)) {
            is ABSegmentGoal.ID -> ABSegmentGoal(goalID, time.days)
            is AudienceGoal.ID -> AudienceGoal(goalID)
            is AudioGenresGoal.ID -> AudioGenresGoal(goalID)
            is BehaviorsGoal.ID -> BehaviorsGoal(goalID)
            is BrandSafetyGoal.ID -> BrandSafetyGoal(goalID)
            is CDPSegmentGoal.ID -> CDPSegmentGoal(goalID)
            is ContentCategoryGoal.ID -> ContentCategoryGoal(goalID)
            is ContentGenreGoal.ID -> ContentGenreGoal(goalID)
            is ECommereceGoal.ID -> ECommereceGoal(goalID, time.days)
            is FamilyGoal.ID -> FamilyGoal(goalID)
            is HostGoal.ID -> HostGoal(goalID, time.days)
            is InterestsGoal.ID -> InterestsGoal(goalID)
            is InternalGoal.ID -> InternalGoal(goalID)
            is LalSegmentGoal.ID -> LalSegmentGoal(goalID, time.days)
            is MetrikaGoal.ID -> MetrikaGoal(goalID, time.days)
            is MobileAppSpecialGoal.ID -> MobileAppSpecialGoal(goalID, time.days)
            is MobileGoal.ID -> MobileGoal(goalID, time.days)
            is SegmentGoal.ID -> SegmentGoal(goalID)
            is SocialDemoGoal.ID -> SocialDemoGoal(goalID)
            null -> null // Unknown GoalID (negative for ex.)
        }
    }

private fun ConditionJSON.toInterestRule(): InterestsRule {
    require(this.count() == 1) { "ConditionJSON for `interest` type must contain only one rule" }

    val goals = this.first().toGoals()
    require(goals.count() == 1) { "InterestRule must contain only one goal" }

    val goal = goals.first()
    require(goal is InterestsGoal) { "InterestRule must contain only InterestGoal type" }

    return InterestsRule(this.first().type.toDomain(), goal)
}

private fun ConditionJSON.toMetrikaRules() = map { rule ->
    MetrikaRule(rule.type.toDomain(), rule.toGoals().filterIsInstance<MetrikaGoal>())
}

private fun ConditionJSON.toGenericRules() = map { rule ->
    GenericRule(rule.type.toDomain(), rule.toGoals())
}

private fun ResultRow.toRetargetingCondition(): RetargetingCondition {
    val id = this[RetargetingConditions.id]
    val clientId = this[RetargetingConditions.clientId]
    val type = this[RetargetingConditions.type]
    val dbRules = this[RetargetingConditions.conditionJson]

    return when (type) {
        ConditionType.INTERESTS -> InterestsRetargetingCondition(id, clientId, dbRules.toInterestRule())
        ConditionType.METRIKA_GOALS -> MetrikaRetargetingCondition(id, clientId, dbRules.toMetrikaRules())
        ConditionType.DMP -> DMPRetargetingCondition(id, clientId, dbRules.toGenericRules())
        ConditionType.AB_SEGMENTS -> ABSegmentsRetargetingCondition(id, clientId, dbRules.toGenericRules())
        ConditionType.BRANDSAFETY -> BrandSafetyRetargetingCondition(id, clientId, dbRules.toGenericRules())
        ConditionType.GEO_SEGMENTS -> GeoSegmentsRetargetingCondition(id, clientId, dbRules.toGenericRules())
        ConditionType.SHORTCUTS -> ShortcutsRetargetingCondition(id, clientId, dbRules.toGenericRules())
    }
}

class MysqlRetargetingConditionRepository(private val db: ShardedDatabase) : RetargetingConditionRepository {
    override fun create(retargetingCondition: RetargetingCondition): RetargetingConditionID? {
        val shard = db.shard(retargetingCondition.clientId) ?: return null
        val conditionID = db.generateConditionId(retargetingCondition.clientId) ?: return null

        transaction(shard) {
            RetargetingConditions.insert {
                it[id] = conditionID
                it[clientId] = retargetingCondition.clientId

                when (retargetingCondition) {
                    is ABSegmentsRetargetingCondition -> TODO()
                    is BrandSafetyRetargetingCondition -> TODO()
                    is DMPRetargetingCondition -> TODO()
                    is GeoSegmentsRetargetingCondition -> TODO()
                    is InterestsRetargetingCondition -> {
                        it[type] = ConditionType.INTERESTS
                        it[conditionJson] = listOf(ConditionJSONRule(retargetingCondition.rule))
                    }
                    is MetrikaRetargetingCondition -> {
                        it[type] = ConditionType.METRIKA_GOALS
                        it[name] = retargetingCondition.name
                        it[description] = retargetingCondition.description
                        it[conditionJson] = retargetingCondition.rules.map(::ConditionJSONRule)
                    }
                    is ShortcutsRetargetingCondition -> TODO()
                }

                it[properties] = emptySet()
            }
        }

        return conditionID
    }

    override fun findById(id: RetargetingConditionID): RetargetingCondition? {
        val shard = db.shard(id) ?: return null

        return transaction(shard) {
            addLogger(StdOutSqlLogger)
            RetargetingConditions.select {
                RetargetingConditions.id eq id and
                    not(RetargetingConditions.isDeleted) and
                    RetargetingConditions.properties.notContains(RetargetingConditionProperty.AUTORETARGETING)
            }.singleOrNull()
        }?.toRetargetingCondition()
    }

    override fun findForClient(clientId: ClientID): Collection<RetargetingCondition> {
        val shard = db.shard(clientId) ?: return emptyList()

        return transaction(shard) {
            val query = RetargetingConditions.select {
                RetargetingConditions.clientId eq clientId and
                    not(RetargetingConditions.isDeleted) and
                    RetargetingConditions.properties.notContains(RetargetingConditionProperty.AUTORETARGETING)
            }

            query.mapNotNull(ResultRow::toRetargetingCondition)
        }
    }

    override fun findInterestsForClient(clientId: ClientID): Collection<InterestsRetargetingCondition> {
        val shard = db.shard(clientId) ?: return emptyList()

        return transaction(shard) {
            val query = RetargetingConditions.select {
                RetargetingConditions.clientId eq clientId and
                    not(RetargetingConditions.isDeleted) and
                    RetargetingConditions.properties.contains(RetargetingConditionProperty.INTEREST) and
                    RetargetingConditions.properties.notContains(RetargetingConditionProperty.AUTORETARGETING)
            }

            query.mapNotNull {
                it.toRetargetingCondition() as? InterestsRetargetingCondition
            }
        }
    }
}
