package ru.yandex.direct.infrastructure.mysql.tables

import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import org.jetbrains.exposed.sql.Column
import org.jetbrains.exposed.sql.Table
import org.jetbrains.exposed.sql.javatime.CurrentTimestamp
import org.jetbrains.exposed.sql.javatime.timestamp
import ru.yandex.direct.domain.retargeting.InterestsGoal
import ru.yandex.direct.domain.retargeting.InterestsRule
import ru.yandex.direct.domain.retargeting.MetrikaGoal
import ru.yandex.direct.domain.retargeting.MetrikaRule
import ru.yandex.direct.domain.retargeting.RetargetingConditionProperty
import ru.yandex.direct.domain.retargeting.RuleType
import ru.yandex.direct.infrastructure.mysql.column_types.clientId
import ru.yandex.direct.infrastructure.mysql.column_types.customSet
import ru.yandex.direct.infrastructure.mysql.column_types.json
import ru.yandex.direct.infrastructure.mysql.column_types.retCondId
import ru.yandex.direct.infrastructure.mysql.column_types.safeValueOf

enum class ConditionType {
    INTERESTS,
    METRIKA_GOALS,
    DMP,
    AB_SEGMENTS,
    BRANDSAFETY,
    GEO_SEGMENTS,
    SHORTCUTS
}

@Serializable
data class ConditionJSONGoal(val goal_id: Long, val time: Long) {
    constructor(goal: InterestsGoal) : this(goal_id = goal.id.value, time = goal.membershipLifeSpan.inWholeDays)
    constructor(goal: MetrikaGoal) : this(goal_id = goal.id.value, time = goal.membershipLifeSpan.inWholeDays)
}

@Serializable
enum class ConditionJSONRuleType {
    @SerialName("all")
    ALL,

    @SerialName("not")
    NOT,

    @SerialName("or")
    OR;

    fun toDomain(): RuleType = when (this) {
        ALL -> RuleType.ALL
        NOT -> RuleType.NONE
        OR -> RuleType.ANY
    }

    companion object {
        fun fromDomain(type: RuleType) = when (type) {
            RuleType.ALL -> ALL
            RuleType.NONE -> NOT
            RuleType.ANY -> OR
        }
    }
}

@Serializable
data class ConditionJSONRule(
    val goals: List<ConditionJSONGoal>,
    val type: ConditionJSONRuleType,
    // TODO: interest_type, section_id
) {
    constructor(rule: InterestsRule) : this(
        goals = listOf(ConditionJSONGoal(rule.goal)),
        type = ConditionJSONRuleType.fromDomain(rule.type)
    )

    constructor(rule: MetrikaRule) : this(
        goals = rule.goals.map { ConditionJSONGoal(it) },
        type = ConditionJSONRuleType.fromDomain(rule.type)
    )
}

typealias ConditionJSON = List<ConditionJSONRule>

object RetargetingConditions : Table("retargeting_conditions") {
    var id = retCondId("ret_cond_id")

    var clientId = clientId("ClientID")

    var name = text("condition_name").nullable()

    var description = text("condition_desc").nullable()

    var conditionJson: Column<ConditionJSON> = json("condition_json")

    var isDeleted = bool("is_deleted").default(false)

    var modTime = timestamp("modtime").defaultExpression(CurrentTimestamp())

    var properties = customSet(
        "properties",
        "SET('negative', 'interest', 'autoretargeting')",
        { value ->
            (value as String).split(",").mapNotNull { e -> safeValueOf<RetargetingConditionProperty>(e) }.toSet()
        },
        { s -> s.joinToString(",") { e -> e.name.lowercase() } }
    )

    var type = customEnumeration(
        "retargeting_conditions_type",
        "ENUM('interests','metrika_goals','dmp','ab_segments','brandsafety','geo_segments','shortcuts')",
        { value -> ConditionType.valueOf((value as String).uppercase()) },
        { it.name.lowercase() })

    var reach = integer("reach").nullable()
}
