package ru.yandex.bannerstorage.harvester.configs;

import javax.sql.DataSource;

import org.apache.commons.dbcp2.BasicDataSource;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Profile;
import org.springframework.core.env.Environment;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.mail.javamail.JavaMailSender;

import ru.yandex.bannerstorage.harvester.controllers.MonitoringController;
import ru.yandex.bannerstorage.harvester.infrastructure.AsyncEmailMessageSender;
import ru.yandex.bannerstorage.harvester.infrastructure.MonitoringProvider;
import ru.yandex.bannerstorage.harvester.integration.direct.DirectClient;
import ru.yandex.bannerstorage.harvester.integration.stillage.StillageClient;
import ru.yandex.bannerstorage.harvester.utils.RestTemplateFactory;

/**
 * Глобальный spring-овский конфиг
 *
 * @author egorovmv
 */
@Configuration
@Lazy(false)
public class GlobalConfig {
    @Bean(destroyMethod = "close")
    public BasicDataSource dataSource(
            @NotNull @Value("${jdbc.driverClassName}") String driverClassName,
            @NotNull @Value("${bannerstorage.db.url}") String url,
            @NotNull @Value("${bannerstorage.db.userName}") String userName,
            @NotNull @Value("${bannerstorage.db.password}") String password,
            @NotNull @Value("${jdbc.validationQuery}") String validationQuery,
            @Value("${jdbc.maxTotal}") int maxTotal,
            @Value("${jdbc.maxIdle}") int maxIdle,
            @Value("${jdbc.removeAbandonedTimeout}") int removeAbandonedTimeout) {
        BasicDataSource dataSource = new BasicDataSource();
        dataSource.setDriverClassName(driverClassName);
        dataSource.setUrl(url);
        dataSource.setUsername(userName);
        dataSource.setPassword(password);
        dataSource.setValidationQuery(validationQuery);
        dataSource.setMaxTotal(maxTotal);
        dataSource.setMaxIdle(maxIdle);
        dataSource.setRemoveAbandonedTimeout(removeAbandonedTimeout);
        return dataSource;
    }

    @Bean
    public JdbcTemplate jdbcTemplate(@NotNull DataSource dataSource) {
        return new JdbcTemplate(dataSource);
    }

    @Bean
    public NamedParameterJdbcTemplate namedParameterJdbcTemplate(@NotNull DataSource dataSource) {
        return new NamedParameterJdbcTemplate(dataSource);
    }

    @Bean
    public DataSourceTransactionManager transactionManager(@NotNull DataSource dataSource) {
        return new DataSourceTransactionManager(dataSource);
    }

    @Bean
    public MonitoringProvider monitoringProvider(@NotNull JdbcTemplate jdbcTemplate) {
        return new MonitoringProvider(jdbcTemplate);
    }

    @Bean
    public MonitoringController monitoringController(@NotNull MonitoringProvider monitoringProvider) {
        return new MonitoringController(monitoringProvider);
    }

    @Bean
    public RestTemplateFactory restTemplateFactory(
            @NotNull Environment environment,
            @Value("${harvester.restTemplateFactory.defaultMaxConnPerRoute:0}") int defaultMaxConnPerRoute,
            @Value("${harvester.restTemplateFactory.defaultMaxConnTotal:0}") int defaultMaxConnTotal) {
        return new RestTemplateFactory(
                defaultMaxConnPerRoute,
                defaultMaxConnTotal,
                environment.acceptsProfiles("ssl_insecure"));
    }

    @Bean
    public StillageClient stillageClient(@NotNull @Value("${stillage.url}") String url) {
        return new StillageClient(url);
    }

    @Bean
    public DirectClient directClient(@NotNull @Value("${direct.intapi.url}") String url) {
        return new DirectClient(url);
    }

    @Bean
    @Profile("testing")
    public AsyncEmailMessageSender testingEmailMessageSender() {
        return new AsyncEmailMessageSender.FakeAsyncEmailMessageSender();
    }

    @Bean
    @Profile("production")
    public AsyncEmailMessageSender productionEmailMessageSender(
            @NotNull JavaMailSender mailSender,
            @NotNull @Value("${bannerstorage.harvester.emailMessageSender.from}") String from,
            @Value("${bannerstorage.harvester.emailMessageSender.maxPoolSize:1}") int maxPoolSize) {
        return new AsyncEmailMessageSender.JavaAsyncEmailMessageSender(mailSender, from, maxPoolSize);
    }
}
