package ru.yandex.bannerstorage.harvester.configs.queues;

import java.util.Arrays;
import java.util.stream.Collectors;

import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Profile;
import org.springframework.core.env.Environment;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.web.client.RestTemplate;

import ru.yandex.bannerstorage.harvester.infrastructure.AsyncEmailMessageSender;
import ru.yandex.bannerstorage.harvester.infrastructure.PassportAuthenticator;
import ru.yandex.bannerstorage.harvester.queues.automoderation.AutoModerationNotifyQueueObserver;
import ru.yandex.bannerstorage.harvester.queues.automoderation.AutoModerationStartQueueObserver;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.CreativeService;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.CreativeWorkflowService;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.virustotal.VirusTotalClient;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.virustotal.VirusTotalClientFactory;
import ru.yandex.bannerstorage.harvester.utils.SslUtils;
import ru.yandex.bannerstorage.messaging.services.QueueMessageSender;

/**
 * Spring config для автомодерации
 *
 * @author egorovmv
 */
@Configuration
@Profile("automoderation")
@Lazy(false)
public class AutoModerationConfig {
    @Bean
    public CreativeService creativeService(
            @NotNull JdbcTemplate jdbcTemplate,
            CloseableHttpClient httpClient,
            @Value("${bannerstorage.harvester.autoModeration.creativeService.cacheTimeoutInSec:600}") int cacheTimeoutInSec) {
        return new CreativeService(jdbcTemplate, httpClient, cacheTimeoutInSec);
    }

    @Bean
    public CreativeWorkflowService creativeWorkflowService(
            @NotNull CreativeService creativeService,
            @NotNull QueueMessageSender queueMessageSender,
            @NotNull AsyncEmailMessageSender emailMessageSender,
            @NotNull @Value("${bannerstorage.harvester.autoModeration.rejectRecipients}") String rejectRecipients,
            @NotNull @Value("${premoderation.url:https://premoderation.mediaselling.yandex.net}") String premoderationUrl) {
        return new CreativeWorkflowService(
                creativeService,
                queueMessageSender,
                emailMessageSender,
                Arrays.stream(rejectRecipients.split(","))
                        .map(String::trim)
                        .toArray(String[]::new),
                premoderationUrl);
    }

    @Bean
    public CloseableHttpClient httpClient(Environment environment) {
        if (environment.acceptsProfiles("ssl_insecure")) {
            return HttpClientBuilder.create().useSystemProperties()
                    .setSSLContext(SslUtils.buildInsecureSSLContext())
                    .setSSLHostnameVerifier(SslUtils.buildInsecureHostnameVerifier())
                    .setMaxConnPerRoute(100)
                    .setMaxConnTotal(100)
                    .build();
        } else {
            return HttpClientBuilder.create().useSystemProperties()
                    .setMaxConnPerRoute(100)
                    .setMaxConnTotal(100)
                    .build();
        }
    }

    @Bean
    public RestTemplate restTemplate(
            CloseableHttpClient httpClient,
            @Value("${automoderation.connectTimeoutInMS:60000}") int connectTimeoutInMS,
            @Value("${automoderation.readTimeoutInMS:60000}") int readTimeoutInMS) {
        HttpComponentsClientHttpRequestFactory requestFactory =
                new HttpComponentsClientHttpRequestFactory(httpClient);
        requestFactory.setConnectTimeout(connectTimeoutInMS);
        requestFactory.setReadTimeout(readTimeoutInMS);
        return new RestTemplate(requestFactory);
    }

//    @Bean
//    public VirusTotalStartQueueObserver virusTotalStartQueueObserver(
//            @NotNull VirusTotalClient virusTotalClient,
//            @NotNull JdbcTemplate jdbcTemplate,
//            @Value("${bannerstorage.harvester.virusTotalStartQueue.pollInternalInMS:1000}") int pollIntervalInMS
//    ) {
//        return new VirusTotalStartQueueObserver(
//                virusTotalClient,
//                jdbcTemplate,
//                pollIntervalInMS
//        );
//    }

    @Bean
    public AutoModerationStartQueueObserver autoModerationStartQueueObserver(
            @Value("${phantomjs.service.url}") @NotNull String phantomJsServiceURL,
            @NotNull CreativeWorkflowService creativeWorkflowService,
            @NotNull CreativeService creativeService,
            @NotNull QueueMessageSender messageSender,
            @NotNull @Value("${phantomjs.trustedHosts:}") String trustedHosts,
            @NotNull Environment environment,
            @Value("${bannerstorage.harvester.autoModerationStartQueue.pollInternalInMS:1000}") int pollIntervalInMS) {
        return new AutoModerationStartQueueObserver(
                phantomJsServiceURL,
                creativeService,
                creativeWorkflowService,
                messageSender,
                Arrays.stream(trustedHosts.split(","))
                        .map(String::trim)
                        .filter(s -> !s.isEmpty())
                        .collect(Collectors.toSet()),
                environment.acceptsProfiles("ssl_insecure"),
                pollIntervalInMS);
    }

    @Bean
    public PassportAuthenticator passportAuthenticator(
            @NotNull @Value("${passport.url}") String serviceUrl,
            @Value("${passport.connectTimeoutInMS:30000}") int connectTimeoutInMS,
            @Value("${passport.readTimeoutInMS:30000}") int readTimeoutInMS) {
        return new PassportAuthenticator(serviceUrl, connectTimeoutInMS, readTimeoutInMS);
    }

    @Bean
    public VirusTotalClient virusTotalClient(
            @Value("${virustotal.url}") String url,
            @NotNull PassportAuthenticator passportAuthenticator,
            @Value("${virustotal.robotLogin}") String robotLogin,
            @Value("${virustotal.robotPassword}") String robotPassword,
            @Value("${virustotal.renewSessionTimeoutInMin:30}") int renewSessionTimeoutInMin,
            @Value("${virustotal.connectTimeoutInMS:30000}") int connectTimeoutInMS,
            @Value("${virustotal.readTimeoutInMS:30000}") int readTimeoutInMS) {
        return VirusTotalClientFactory.newInstance(
                url,
                passportAuthenticator,
                robotLogin,
                robotPassword,
                renewSessionTimeoutInMin,
                connectTimeoutInMS,
                readTimeoutInMS);
    }

//    @Bean
//    public VirusTotalPollQueueObserver virusTotalPollQueueObserver(
//            @NotNull BasicDataSource basicDataSource,
//            @NotNull VirusTotalClient virusTotalClient,
//            @NotNull AsyncEmailMessageSender emailMessageSender,
//            @NotNull @Value("${bannerstorage.harvester.autoModeration.rejectRecipients}") String recipients,
//            @NotNull @Value("${premoderation.url:https://premoderation.mediaselling.yandex.net}") String premoderationUrl,
//            @Value("${bannerstorage.harvester.virusTotalPollQueue.pollInternalInMS:10000}") int pollIntervalInMS
//    ) {
//        return new VirusTotalPollQueueObserver(
//                basicDataSource,
//                virusTotalClient,
//                emailMessageSender,
//                Arrays.stream(recipients.split(",")).map(String::trim).toArray(String[]::new),
//                premoderationUrl,
//                pollIntervalInMS
//        );
//    }

    @Bean
    public AutoModerationNotifyQueueObserver autoModerationNotifyQueueObserver(
            @NotNull CreativeService creativeService,
            @Value("${bannerstorage.harvester.autoModerationNotifyQueueObserver.pollInternalInMS:1000}") int pollIntervalInMS) {
        return new AutoModerationNotifyQueueObserver(creativeService, pollIntervalInMS);
    }
}
