package ru.yandex.bannerstorage.harvester.configs.queues;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Profile;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;

import ru.yandex.bannerstorage.harvester.controllers.SettingsController;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.CreativeChangedQueueObserver;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.services.CreativeService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.services.impl.JdbcCreativeService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbClientService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbIntegrationHealthService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.impl.BsApiClientService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.impl.HttpRtbClientService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.impl.JdbcRtbIntegrationHealthService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.RecvUnmoderatedAssembliesQueueObserver;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.SendModeratedOffersQueryObserver;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.services.AssemblyService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.services.impl.JdbcAssemblyService;
import ru.yandex.bannerstorage.harvester.utils.RestTemplateFactory;
import ru.yandex.direct.bs.dspcreative.configuration.BsDspCreativeConfiguration;
import ru.yandex.direct.bs.dspcreative.service.DspCreativeYtExporter;

/**
 * @author egorovmv
 */
@Configuration
@Import({BsDspCreativeConfiguration.class})
@Lazy(false)
public class RtbIntegrationConfig {
    @RtbCreativeChangedProfile
    @RtbPostModerationProfile
    @Bean
    public RtbClientService rtbClientService(
            @NotNull RestTemplateFactory restTemplateFactory,
            @NotNull @Value("${rtbHost.url}") String serviceUrl,
            @Value("${rtbHost.connectTimeoutInMS:60000}") int connectTimeoutInMS,
            @Value("${rtbHost.readTimeoutInMS:60000}") int readTimeoutInMS) {
        return new HttpRtbClientService(
                restTemplateFactory, serviceUrl, connectTimeoutInMS, readTimeoutInMS);
    }

    @RtbCreativeChangedProfile
    @RtbPostModerationProfile
    @Bean
    public RtbIntegrationHealthService rtbIntegrationHealthService(@NotNull JdbcTemplate jdbcTemplate) {
        return new JdbcRtbIntegrationHealthService(jdbcTemplate);
    }

    @RtbCreativeChangedProfile
    @Bean
    public CreativeService rtbCreativeService(@NotNull JdbcTemplate jdbcTemplate) {
        return new JdbcCreativeService(jdbcTemplate);
    }

    @RtbCreativeChangedProfile
    @Bean
    public BsApiClientService bsApiClientService(
            @NotNull RestTemplateFactory restTemplateFactory,
            @NotNull @Value("${bsApi.url:https://rest-api.bannerstorage.yandex.net}") String bsApiUrl) {
        return new BsApiClientService(restTemplateFactory, bsApiUrl);
    }

    @RtbCreativeChangedProfile
    @Bean
    public CreativeChangedQueueObserver creativeChangedQueueObserver(
            @NotNull CreativeService creativeService,
            @NotNull RtbClientService rtbClientService,
            @NotNull RtbIntegrationHealthService healthService,
            @NotNull BsApiClientService bsApiClientService,
            @NotNull DspCreativeYtExporter dspCreativeYtExporter) {
        return new CreativeChangedQueueObserver(creativeService, rtbClientService, healthService, bsApiClientService,
                dspCreativeYtExporter);
    }

    @RtbPostModerationProfile
    @Bean
    public AssemblyService assemblyService(@NotNull NamedParameterJdbcTemplate jdbcTemplate) {
        return new JdbcAssemblyService(jdbcTemplate);
    }

    @RtbPostModerationProfile
    @Bean
    public SendModeratedOffersQueryObserver sendModeratedOffersQueryObserver(
            @NotNull AssemblyService assemblyService,
            @NotNull RtbClientService rtbClient,
            @NotNull RtbIntegrationHealthService healthService,
            @Value("${rtbHost.postmoderation.sendModeratedOffers.fetchCount:100}") int fetchCount,
            @Value("${rtbHost.postmoderation.sendModeratedOffers.pollInternalInMS:30000}") int pollIntervalInMS) {
        return new SendModeratedOffersQueryObserver(
                assemblyService, rtbClient, healthService, fetchCount, pollIntervalInMS);
    }

    @RtbPostModerationProfile
    @Bean
    public RecvUnmoderatedAssembliesQueueObserver recvUnmoderatedOffersQueueObserver(
            @NotNull AssemblyService assemblyService,
            @NotNull RtbClientService rtbClient,
            @NotNull RtbIntegrationHealthService healthService,
            @Value("${rtbHost.postmoderation.recvUnmoderatedAssemblies.fetchCount:100}") int fetchCount,
            @Value("${rtbHost.postmoderation.recvUnmoderatedAssemblies.pollInternalInMS:30000}") int pollIntervalInMS) {
        return new RecvUnmoderatedAssembliesQueueObserver(
                assemblyService, rtbClient, healthService, fetchCount, pollIntervalInMS);
    }

    @RtbCreativeChangedProfile
    @RtbPostModerationProfile
    @Bean
    public SettingsController settingsController(@NotNull RtbClientService rtbClientService) {
        return new SettingsController(rtbClientService);
    }

    @Target(ElementType.METHOD)
    @Retention(RetentionPolicy.RUNTIME)
    @Profile("rtb_creative_changed")
    @interface RtbCreativeChangedProfile {
    }

    @Target(ElementType.METHOD)
    @Retention(RetentionPolicy.RUNTIME)
    @Profile("rtb_postmoderation")
    @interface RtbPostModerationProfile {
    }
}
