package ru.yandex.bannerstorage.harvester.infrastructure;

import java.io.Closeable;
import java.util.Arrays;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.validation.constraints.NotNull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;

/**
 * @author egorovmv
 */
public interface AsyncEmailMessageSender extends Closeable {
    void sendMessage(@NotNull String[] recipients, @NotNull String subject, @NotNull String message);

    default void close() {
    }

    class FakeAsyncEmailMessageSender implements AsyncEmailMessageSender {
        @Override
        public void sendMessage(
                @NotNull String[] recipients, @NotNull String subject, @NotNull String message) {
        }
    }

    class JavaAsyncEmailMessageSender implements AsyncEmailMessageSender {
        private static final Logger logger = LoggerFactory.getLogger(JavaAsyncEmailMessageSender.class);

        private final JavaMailSender mailSender;
        private final String from;
        private final ExecutorService sendMailExecutor;

        public JavaAsyncEmailMessageSender(
                @NotNull JavaMailSender mailSender,
                @NotNull String from,
                int maxPoolSize) {
            Objects.requireNonNull(mailSender, "mailSender");
            Objects.requireNonNull(from, "from");
            if (maxPoolSize <= 0)
                throw new IllegalArgumentException("maxPoolSize: " + maxPoolSize);
            this.mailSender = Objects.requireNonNull(mailSender, "mailSender");
            this.from = from;
            this.sendMailExecutor = new ThreadPoolExecutor(
                    1, maxPoolSize, 5, TimeUnit.MINUTES, new LinkedBlockingDeque<>());
        }

        @Override
        public void sendMessage(
                @NotNull String[] recipients, @NotNull String subject, @NotNull String message) {
            if (recipients.length == 0)
                return;
            sendMailExecutor.execute(() -> {
                try {
                    logger.trace("Sending email message \"{}\" to \"{}\" ()...", subject, Arrays.asList(recipients));
                    SimpleMailMessage mailMessage = new SimpleMailMessage();
                    mailMessage.setFrom(from);
                    mailMessage.setTo(recipients);
                    mailMessage.setSubject(subject);
                    mailMessage.setText(message);
                    mailSender.send(mailMessage);
                    logger.trace("Email message \"{}\" to \"{}\" sent", subject, Arrays.asList(recipients));
                } catch (Exception e) {
                    logger.error(
                            String.format(
                                    "Can't send email message \"%s\" to \"%s",
                                    subject,
                                    Arrays.asList(recipients)), e);
                }
            });
        }

        public void close() {
            sendMailExecutor.shutdown();
            try {
                sendMailExecutor.awaitTermination(Integer.MAX_VALUE, TimeUnit.SECONDS);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }
}
