package ru.yandex.bannerstorage.harvester.infrastructure;

import java.net.URI;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

/**
 * @author egorovmv
 */
public final class PassportAuthenticator {
    public static final String YANDEX_UID_COOKIE = "yandexuid";
    public static final String YANDEX_LOGIN_COOKIE = "yandex_login";
    public static final String YANDEX_SESSIONID_COOKIE = "Session_id";
    public static final String YANDEX_SESSIONID2_COOKIE = "sessionid2";

    private static final Logger logger = LoggerFactory.getLogger(PassportAuthenticator.class);

    private static final Set<String> requiredPassportCookieNames = Sets.newHashSet(
            YANDEX_UID_COOKIE, YANDEX_LOGIN_COOKIE, YANDEX_SESSIONID_COOKIE, YANDEX_SESSIONID2_COOKIE);

    private final RestTemplate restTemplate;
    private final URI serviceUrl;

    public PassportAuthenticator(
            @NotNull String serviceUrl,
            int connectTimeoutInMs,
            int readTimeoutInMs) {
        Objects.requireNonNull(serviceUrl, "serviceUrl");
        if (connectTimeoutInMs <= 0)
            throw new IllegalArgumentException("connectTimeoutInMs: " + connectTimeoutInMs);
        if (readTimeoutInMs <= 0)
            throw new IllegalArgumentException("readTimeoutInMs: " + readTimeoutInMs);

        HttpComponentsClientHttpRequestFactory requestFactory = new HttpComponentsClientHttpRequestFactory();
        requestFactory.setConnectTimeout(connectTimeoutInMs);
        requestFactory.setReadTimeout(readTimeoutInMs);
        this.restTemplate = new RestTemplate(requestFactory);

        this.serviceUrl = UriComponentsBuilder.fromHttpUrl(serviceUrl)
                .pathSegment("auth")
                .build()
                .toUri();
    }

    public PassportSession authenticate(@NotNull String login, @NotNull String password) {
        Objects.requireNonNull(login, "login");
        Objects.requireNonNull(login, "password");

        UUID requestId = UUID.randomUUID();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);

        MultiValueMap<String, Object> requestBody = new LinkedMultiValueMap<>();
        requestBody.add("login", login);
        requestBody.add("passwd", password);

        try {
            logger.info(
                    "Authenticating (RequestId: {}, PassportUrl: {}, Login: {})...",
                    requestId, serviceUrl, login);

            ResponseEntity<String> responseEntity = restTemplate.exchange(
                    serviceUrl,
                    HttpMethod.POST,
                    new HttpEntity<>(requestBody, headers),
                    String.class);

            Map<String, String> passportCookies = responseEntity.getHeaders()
                    .get(HttpHeaders.SET_COOKIE)
                    .stream()
                    .map(Cookie::parse)
                    .filter(c -> requiredPassportCookieNames.contains(c.getName()))
                    .collect(Collectors.toMap(Cookie::getName, Cookie::getValue));

            logger.info("Authenticated (RequestId: {}, Response: {})", requestId, passportCookies);

            Set<String> missedCookieNames = Sets.difference(requiredPassportCookieNames, passportCookies.keySet());
            if (!missedCookieNames.isEmpty())
                throw new PassportAuthenticationException("Missed required cookies: " + missedCookieNames.toString());

            return new PassportSession(
                    passportCookies.get(YANDEX_UID_COOKIE),
                    passportCookies.get(YANDEX_LOGIN_COOKIE),
                    passportCookies.get(YANDEX_SESSIONID_COOKIE),
                    passportCookies.get(YANDEX_SESSIONID2_COOKIE));
        } catch (RestClientException e) {
            throw new PassportAuthenticationException(e);
        }
    }

    private static final class Cookie {
        private final String name;
        private final String value;

        public Cookie(@NotNull String name, @NotNull String value) {
            this.name = Objects.requireNonNull(name);
            this.value = Objects.requireNonNull(value);
        }

        public static Cookie parse(@NotNull String cookieValue) {
            Objects.requireNonNull(cookieValue, "cookieValue");
            String[] nameValue = cookieValue.substring(0, cookieValue.indexOf(';')).split("=");
            return new Cookie(nameValue[0].trim(), nameValue[1].trim());
        }

        @Override
        public String toString() {
            return "Cookie{" +
                    "name='" + name + '\'' +
                    ", value='" + value + '\'' +
                    '}';
        }

        public String getName() {
            return name;
        }

        public String getValue() {
            return value;
        }
    }
}
