package ru.yandex.bannerstorage.harvester.integration.direct;

import java.io.UncheckedIOException;
import java.net.URI;
import java.util.List;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

public class DirectClient {
    private static final Logger logger = LoggerFactory.getLogger(DirectClient.class);

    private final static String PATH = "/jsonrpc/BsFront";

    private static final int CONNECT_TIMEOUT = 5_000;
    private static final int READ_TIMEOUT = 10_000;

    private final RestTemplate restTemplate;
    private final String serviceUrl;
    private final ObjectMapper jsonMapper;

    public DirectClient(String serviceUrl) {
        this.serviceUrl = serviceUrl;

        this.jsonMapper = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

        HttpComponentsClientHttpRequestFactory requestFactory = new HttpComponentsClientHttpRequestFactory();
        requestFactory.setConnectTimeout(CONNECT_TIMEOUT);
        requestFactory.setReadTimeout(READ_TIMEOUT);
        this.restTemplate = new RestTemplate(requestFactory);
    }

    public void notifyCreativeChanged(List<Integer> creativeIds) {
        if (null == creativeIds || creativeIds.isEmpty())
            throw new IllegalArgumentException("creativeIds shouldn't be empty");

        URI requestUri = UriComponentsBuilder.fromHttpUrl(serviceUrl).path(PATH).build().toUri();
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        String body;
        try {
            body = jsonMapper.writeValueAsString(new CreativeChangedRequest(creativeIds));
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }

        logger.info("Notifying Direct about changed creatives: {}", body);

        try {
            restTemplate.postForObject(requestUri, new HttpEntity<>(body, headers), Void.class);
        } catch (RestClientException e) {
            logger.error("Can't notify Direct about changed creatives", e);
            throw e;
        }
    }
}
