package ru.yandex.bannerstorage.harvester.queues;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.time.Duration;
import java.time.Instant;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.util.stream.Collectors.toList;

public class TaskQueueObserver {
    private static final Logger logger = LoggerFactory.getLogger(TaskQueueObserver.class);

    private final String table;
    private final String taskType;
    private final UUID workerId;
    private final Thread thread;
    private final ITaskProcessor processor;
    private final TaskRepository taskRepository;
    private final int batchSize;

    private enum State {
        NotStarted,
        Started,
        Stopped
    }

    private State state = State.NotStarted;

    private static final Duration WAIT_ON_EMPTY_QUEUE_INTERVAL = Duration.ofSeconds(60);

    public TaskQueueObserver(String table, String taskType,
                             ITaskProcessor processor,
                             TaskRepository taskRepository,
                             int batchSize) {
        this.table = table;
        this.taskType = taskType;
        this.workerId = UUID.randomUUID();
        this.processor = processor;
        this.taskRepository = taskRepository;
        this.batchSize = batchSize;
        this.thread = new Thread(() -> {
            while (!Thread.interrupted()) {
                try {
                    List<Task> tasks = taskRepository.lockAndGetNextTasks(table, taskType, workerId, batchSize);
                    if (tasks.isEmpty()) {
                        logger.info("No new tasks retrieved, waiting for {} seconds",
                                WAIT_ON_EMPTY_QUEUE_INTERVAL.toMillis() / 1000);
                        sleep(WAIT_ON_EMPTY_QUEUE_INTERVAL);
                    } else {
                        try {
                            processor.process((List) tasks);
                            taskRepository.deleteTasks(table, workerId,
                                    tasks.stream().map(Task::getId).collect(toList()));
                        } catch (RuntimeException e) {
                            logger.error("Error while tasks processing, tasks will be rescheduled", e);
                            //
                            String lastError = extractStacktrace(e);
                            Instant now = Instant.now();
                            for (Task task : tasks) {
                                task.setErrorsCount(task.getErrorsCount() + 1);
                                task.setLastError(lastError);
                                task.setLastErrorTime(new Date(now.toEpochMilli()));
                                Instant nextTime = now.plusSeconds((long) (60 * Math.pow(1.5, task.getErrorsCount())));
                                task.setNextTryTime(new Date(nextTime.toEpochMilli()));
                            }
                            taskRepository.updateTasks(table, workerId, tasks);
                        }
                    }
                } catch (Exception e) {
                    logger.error("Unhandled exception, thread will continue after delay", e);
                    sleep(Duration.ofMinutes(1));
                } catch (Throwable e) {
                    logger.error("Unhandled throwable, thread will stop", e);
                    break;
                }
            }
        });
    }

    private String extractStacktrace(Exception exception) {
        try (StringWriter writer = new StringWriter()) {
            try (PrintWriter printWriter = new PrintWriter(writer)) {
                exception.printStackTrace(printWriter);
                printWriter.flush();
                return writer.toString();
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void sleep(Duration duration) {
        try {
            Thread.sleep(duration.toMillis());
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
    }

    public void start() {
        Preconditions.checkState(state == State.NotStarted);
        thread.start();
        state = State.Started;
    }

    public void stop() throws InterruptedException {
        Preconditions.checkState(state == State.Started);
        thread.interrupt();
        thread.join(); // IS-NOT-COMPLETABLE-FUTURE-JOIN
        state = State.Stopped;
    }
}
