package ru.yandex.bannerstorage.harvester.queues.automoderation;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.jetbrains.annotations.NotNull;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import ru.yandex.bannerstorage.harvester.queues.automoderation.models.AutoModerationStartRequest;
import ru.yandex.bannerstorage.harvester.queues.automoderation.models.VirusTotalStartRequest;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.CreativeService;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.CreativeWorkflowService;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.CheckResult;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.CheckResultStatus;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.Creative;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.TaskDefinition;
import ru.yandex.bannerstorage.harvester.utils.SslUtils;
import ru.yandex.bannerstorage.messaging.services.QueueMessageOnErrorStrategy;
import ru.yandex.bannerstorage.messaging.services.QueueMessageSender;
import ru.yandex.bannerstorage.messaging.services.SimpleOneWayQueueObserver;

import static java.util.Collections.singletonList;
import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;

/**
 * Очереди отправки версий креативов на автомодерацию
 *
 * @author egorovmv
 */
public final class AutoModerationStartQueueObserver extends SimpleOneWayQueueObserver<AutoModerationStartRequest> {
    private static final String QUEUE_ID = "dbo.AutoModerationStartServiceQueue";

    private final CreativeService creativeService;
    private final CreativeWorkflowService creativeWorkflowService;
    private final QueueMessageSender messageSender;

    // Проверки, выполняемые с помощью Phantom JS
    private static final int JS_ERRORS_NMB = 1;
    private static final int EXTERNAL_REQUEST_NMB = 2;
    private static final String EXTERNAL_REQUESTS = "external_requests";
    private static final String JS_ERRORS = "js_errors";

    private final String phantomJsServiceUrl;

    private final ObjectMapper objectMapper;
    private final boolean sslInsecure;

    private static final Set<String> TYPES_TO_LOG =
            Sets.newHashSet("phantom_error", "page_error", "resource_load_error", "load_error", "runtime_error");
    private final List<Predicate<String>> trustedHosts;

    private static final int MOBILE_STORE_LINK_NMB = 4;

    private static final String VIRUSTOTAL = "virus_total";

    public AutoModerationStartQueueObserver(
            @NotNull String phantomJsServiceUrl,
            @NotNull CreativeService creativeService,
            @NotNull CreativeWorkflowService creativeWorkflowService,
            @NotNull QueueMessageSender messageSender,
            @NotNull Set<String> trustedHosts,
            boolean sslInsecure,
            int pollIntervalInMS) {
        super(
                QUEUE_ID,
                pollIntervalInMS,
                5,
                5,
                QueueMessageOnErrorStrategy.INFINITE_POISON_START_NEW_SESSION,
                AutoModerationStartRequest.class);
        this.phantomJsServiceUrl = phantomJsServiceUrl;
        this.creativeService = creativeService;
        this.creativeWorkflowService = creativeWorkflowService;
        this.messageSender = messageSender;
        this.sslInsecure = sslInsecure;
        this.objectMapper = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        this.trustedHosts = trustedHosts.stream()
                .map(h -> Pattern.compile("^(http|https)://" + h + "/", Pattern.CASE_INSENSITIVE).asPredicate())
                .collect(toList());
    }

    private CloseableHttpClient buildHttpClient() {
        if (sslInsecure) {
            return HttpClientBuilder.create().useSystemProperties()
                    .setSSLContext(SslUtils.buildInsecureSSLContext())
                    .setSSLHostnameVerifier(SslUtils.buildInsecureHostnameVerifier())
                    .setMaxConnPerRoute(100)
                    .setMaxConnTotal(100)
                    .build();
        } else {
            return HttpClientBuilder.create().useSystemProperties()
                    .setMaxConnPerRoute(100)
                    .setMaxConnTotal(100)
                    .build();
        }
    }

    private RestTemplate buildRestTemplate() {
        HttpComponentsClientHttpRequestFactory requestFactory =
                new HttpComponentsClientHttpRequestFactory(buildHttpClient());
        requestFactory.setConnectTimeout(1_000);
        requestFactory.setReadTimeout(60_000);
        RestTemplate restTemplate = new RestTemplate(requestFactory);
        // Устанавливаем в качестве message converters только один конвертер для json,
        // чтобы не использовались другие (иначе может быть выбран конвертер для xml, к примеру)
        restTemplate.setMessageConverters(singletonList(new MappingJackson2HttpMessageConverter()));
        return restTemplate;
    }

    @Override
    protected void doProcessMessage(@NotNull AutoModerationStartRequest message) {
        Integer creativeVersionId = message.getCreativeVersionId();

        // Если вообще нет проверок, двигаем креатив сразу в ручную модерацию
        if (message.getRequiredChecks().isEmpty()) {
             creativeService.moveToModeration(creativeVersionId);
            return;
        }

        // Если нужна проверка от Phantom JS, выполняем её сразу
        boolean rejected = false;
        if (message.getRequiredChecks().contains(JS_ERRORS_NMB) ||
                message.getRequiredChecks().contains(EXTERNAL_REQUEST_NMB)) {

            // Зовём ручку и получаем результаты
            List<CheckResult> checkResults = processPhantomJsChecks(
                    creativeVersionId,
                    creativeService.createTaskFor(creativeVersionId, message.getRequiredChecks())
                            .getCreative(),
                    message.getRequiredChecks().contains(JS_ERRORS_NMB),
                    message.getRequiredChecks().contains(EXTERNAL_REQUEST_NMB));

            // Сохраняем результаты проверок
            creativeService.append(creativeVersionId, checkResults);

            List<CheckResult> failedCheckResults = checkResults.stream()
                    .filter(r -> r.getStatus() == CheckResultStatus.FAILED)
                    .collect(Collectors.toList());

            if (!failedCheckResults.isEmpty()) {
                creativeWorkflowService.reject(creativeVersionId, failedCheckResults);
                rejected = true;
            }
        }

        // Если не нужно ждать проверки MOBILE_STORE_LINK, двигаем в модерацию
        if (!rejected && !message.getRequiredChecks().contains(MOBILE_STORE_LINK_NMB)) {
            creativeService.moveToModeration(creativeVersionId);
        }

        // TODO : перетащить сюда код проверки MOBILE_STORE_LINK

        // Сформировать задачу на автомодерацию версии креатива с учетом настроек шаблона
        // (все остальные проверки, кроме phantom js) - тут уже остаётся только вирус тотал
        TaskDefinition taskDefinition = creativeService.createTaskFor(
                creativeVersionId, message.getRequiredChecks());

        // Если нужен вирустотал, то отправляем сообщение в очередь VirusTotalStartServiceQueue
        if (taskDefinition.getChecks().stream().anyMatch(c -> c.getName().equals(VIRUSTOTAL))) {
            messageSender.sendMessage(
                    VirusTotalStartQueueObserver.SERVICE_ID,
                    new VirusTotalStartRequest(creativeVersionId));
        }
    }

    private List<CheckResult> processPhantomJsChecks(int creativeVersionId,
                                                     Creative creative,
                                                     boolean jsErrors,
                                                     boolean externalRequests) {
        RestTemplate restTemplate = buildRestTemplate();
        URL url;
        try {
            // Добавляем в URL номер версии креатива для более простой диагностики PhantomJS API в случае проблем
            url = new URL(new URL(phantomJsServiceUrl), "/check?creative_version_nmb=" + creativeVersionId);
        } catch (MalformedURLException e) {
            throw new IllegalStateException(e);
        }

        ZonedDateTime startTime = ZonedDateTime.now();

        ResponseEntity<PhantomJsResponse> responseEntity;
        try {
            getLogger().info("Checking using url '{}' creative: '{}'", url.toString(), toJson(creative));
            responseEntity = restTemplate.postForEntity(url.toString(), creative, PhantomJsResponse.class);
        } catch (RestClientException e) {
            getLogger().error(String.format("Error when checking PhantomJS things (url '%s', creative: '%s')",
                    url.toString(), toJson(creative)), e);
            throw e;
        }

        ZonedDateTime finishTime = ZonedDateTime.now();

        if (200 != responseEntity.getStatusCodeValue()) {
            getLogger().error("Phantom JS service returned with status code {}. Response: {}",
                    responseEntity.getStatusCodeValue(),
                    responseEntity.getBody());
            throw new RuntimeException("Phantom JS check error");
        }

        List<LogMessage> logMessages = responseEntity.getBody().getOutput();
        Map<String, List<LogMessage>> messageGroups = logMessages.stream()
                .collect(groupingBy(LogMessage::getType,
                        collectingAndThen(toList(),
                                l -> l.stream().sorted(comparing(LogMessage::getElapsed)).collect(toList()))));

        for (LogMessage logMessage : logMessages) {
            if (TYPES_TO_LOG.contains(logMessage.getType())) {
                if (logMessage.getStacktrace() != null) {
                    getLogger().error("Error type: {} with message: {}, reason: {}, stacktrace: '{}'", logMessage.getType(),
                            logMessage.getMessage(), logMessage.getReason(), logMessage.getStacktrace());
                } else {
                    getLogger().error("Error type: {} with message: {}", logMessage.getType(), logMessage.getMessage());
                }
            }
        }

        List<CheckResult> results = new ArrayList<>();

        if (jsErrors) {
            final Set<String> jsErrorMessageTypes = Sets.newHashSet("phantom_error", "page_error");
            final Set<String> resourceLoadErrorMessageTypes = Sets.newHashSet("resource_load_error", "load_error");

            List<String> errors = new ArrayList<>();
            errors.addAll(messageGroups.entrySet().stream()
                    .filter(entry -> jsErrorMessageTypes.contains(entry.getKey()))
                    .flatMap(entry -> entry.getValue().stream())
                    .map((logMessage) -> "Ошибка: " + logMessage.getMessage())
                    .collect(toList()));

            errors.addAll(messageGroups.entrySet().stream()
                    .filter(entry -> resourceLoadErrorMessageTypes.contains(entry.getKey()))
                    .flatMap(entry -> entry.getValue().stream())
                    .map((logMessage) -> "Ошибка при загрузке: " + logMessage.getMessage())
                    .collect(toList()));

            results.add(
                    new CheckResult(
                            JS_ERRORS,
                            Collections.emptyMap(),
                            errors.isEmpty() ? CheckResultStatus.PASSED : CheckResultStatus.FAILED,
                            startTime,
                            finishTime,
                            errors.stream().map(errorMessage -> ImmutableMap.<String, Object>of("value", errorMessage)).collect(toList())
                    ));
        }

        if (externalRequests) {
            Stream<LogMessage> externalRequestMessages = messageGroups.getOrDefault(
                    "external_request", Collections.emptyList())
                    .stream();

            if (!trustedHosts.isEmpty()) {
                externalRequestMessages = externalRequestMessages.filter(
                        m -> trustedHosts.stream().noneMatch(h -> h.test(m.getMessage())));
            }

            List<String> errors = externalRequestMessages
                    .map((logMessage) -> "Внешнее обращение: " + logMessage.getMessage())
                    .collect(toList());

            results.add(
                    new CheckResult(
                            EXTERNAL_REQUESTS,
                            Collections.emptyMap(),
                            errors.isEmpty() ? CheckResultStatus.PASSED : CheckResultStatus.FAILED,
                            startTime,
                            finishTime,
                            errors.stream().map(errorMessage -> ImmutableMap.<String, Object>of("value", errorMessage)).collect(toList())
                    ));
        }

        return results;
    }

    private static class PhantomJsResponse {
        private List<LogMessage> output;
        private String error;

        public List<LogMessage> getOutput() {
            return output;
        }

        public void setOutput(List<LogMessage> output) {
            this.output = output;
        }

        public String getError() {
            return error;
        }

        public void setError(String error) {
            this.error = error;
        }
    }

    private static class LogMessage {
        private int elapsed;
        private String type;
        private String message;
        private String reason;
        private String stacktrace;
        private List<TraceInfo> trace;
        private int status;

        public int getElapsed() {
            return elapsed;
        }

        public void setElapsed(int elapsed) {
            this.elapsed = elapsed;
        }

        public String getType() {
            return type;
        }

        public void setType(String type) {
            this.type = type;
        }

        public String getMessage() {
            return message;
        }

        public String getReason() {
            return reason;
        }

        public String getStacktrace() {
            return stacktrace;
        }

        public void setMessage(String message) {
            this.message = message;
        }

        public void setReason(String reason) {
            this.reason = reason;
        }

        public void setStacktrace(String stacktrace) {
            this.stacktrace = stacktrace;
        }

        public List<TraceInfo> getTrace() {
            return trace;
        }

        public void setTrace(List<TraceInfo> trace) {
            this.trace = trace;
        }

        public int getStatus() {
            return status;
        }

        public void setStatus(int status) {
            this.status = status;
        }
    }

    private static class TraceInfo {
        private String file;
        private String line;
        private String function;

        public String getFile() {
            return file;
        }

        public void setFile(String file) {
            this.file = file;
        }

        public String getLine() {
            return line;
        }

        public void setLine(String line) {
            this.line = line;
        }

        public String getFunction() {
            return function;
        }

        public void setFunction(String function) {
            this.function = function;
        }
    }

    public <T> T fromJson(String json, Class<T> clazz) {
        try {
            return objectMapper.readValue(json, clazz);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public String toJson(Object o) {
        try {
            return objectMapper.writeValueAsString(o);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    public <T> T fromJson(String json, TypeReference<T> typeReference) {
        try {
            return objectMapper.readValue(json, typeReference);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}
