package ru.yandex.bannerstorage.harvester.queues.automoderation;

import java.io.UncheckedIOException;
import java.net.URI;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.stream.IntStream;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jetbrains.annotations.NotNull;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCallback;

import ru.yandex.bannerstorage.harvester.queues.automoderation.models.VirusTotalStartRequest;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.virustotal.VirusTotalClient;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.virustotal.VirusTotalSendScanResult;
import ru.yandex.bannerstorage.messaging.services.QueueMessageOnErrorStrategy;
import ru.yandex.bannerstorage.messaging.services.SimpleOneWayQueueObserver;

import static java.util.stream.Collectors.joining;

/**
 * @author elwood
 */
public class VirusTotalStartQueueObserver extends SimpleOneWayQueueObserver<VirusTotalStartRequest> {
    public static final String SERVICE_ID = "VirusTotalStartService";
    private static final String QUEUE_ID = "dbo.VirusTotalStartServiceQueue";

    private final VirusTotalClient virusTotalClient;
    private final JdbcTemplate jdbcTemplate;

    private final ObjectMapper objectMapper;

    // Using time zone = MSK to avoid errors when saving datetimes via JDBC
    private static final String TIMEZONE_MSK = "GMT+3";

    public VirusTotalStartQueueObserver(
            VirusTotalClient virusTotalClient,
            JdbcTemplate jdbcTemplate,
            int pollIntervalInMS) {
        super(
                QUEUE_ID,
                pollIntervalInMS,
                5,
                5,
                QueueMessageOnErrorStrategy.INFINITE_POISON_START_NEW_SESSION,
                VirusTotalStartRequest.class);
        this.virusTotalClient = virusTotalClient;
        this.jdbcTemplate = jdbcTemplate;

        this.objectMapper = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    @Override
    public void doProcessMessage(@NotNull VirusTotalStartRequest message) {
        int creativeVersionId = message.getCreativeVersionId();

        GregorianCalendar sendScanTime = new GregorianCalendar(TimeZone.getTimeZone(TIMEZONE_MSK));
        GregorianCalendar nextAttemptTime = (GregorianCalendar) sendScanTime.clone();
        nextAttemptTime.add(Calendar.MINUTE, 1);

        List<FileInfo> files = getCreativeFiles(creativeVersionId);

        // Посылаем все файлы в вирустотал
        Map<FileInfo, VirusTotalSendScanResult> sendScanResults = new HashMap<>();
        for (FileInfo file : files) {
            try {
                VirusTotalSendScanResult sendScanResult =
                        virusTotalClient.sendForScan(
                                extractFileName(file.getPath()),
                                new URI(file.getContentUrl()));
                sendScanResults.put(file, sendScanResult);
            } catch (Exception e) {
                getLogger().error(
                        "Failed to start virus total scan for file '{}' (creative version {})",
                        file.getContentUrl(),
                        creativeVersionId
                );
                throw new RuntimeException("Failed to start virus total scan", e);
            }
        }

        // Сохраняем полученные айдишники проверок в базу
        String sql =
                "INSERT INTO t_creative_version_virustotal_scan" +
                        "(creative_version_nmb," +
                        " file_instance_nmb," +
                        " send_scan_datetime," +
                        " next_attempt_datetime," +
                        " virustotal_send_scan_result," +
                        " virustotal_resource_id," +
                        " virustotal_permalink," +
                        " virustotal_scan_result," +
                        " virustotal_last_error," +
                        " virustotal_scan_finished," +
                        " virustotal_attempts) VALUES " +
                        IntStream.range(0, sendScanResults.size())
                                .mapToObj(i -> "(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)")
                                .collect(joining(","));
        jdbcTemplate.execute(
                con -> con.prepareStatement(sql),
                (PreparedStatementCallback<Object>) st -> {
                    int baseIndex = 0;
                    for (Map.Entry<FileInfo, VirusTotalSendScanResult> entry : sendScanResults.entrySet()) {
                        FileInfo file = entry.getKey();
                        VirusTotalSendScanResult sendScanResult = entry.getValue();

                        st.setInt(baseIndex + 1, creativeVersionId);
                        st.setInt(baseIndex + 2, file.getFileId());
                        st.setTimestamp(baseIndex + 3,
                                new java.sql.Timestamp(sendScanTime.getTime().getTime()),
                                sendScanTime);
                        st.setTimestamp(baseIndex + 4,
                                new java.sql.Timestamp(nextAttemptTime.getTime().getTime()),
                                nextAttemptTime);
                        st.setString(baseIndex + 5, toJson(sendScanResult));
                        st.setString(baseIndex + 6, sendScanResult.getResource());
                        st.setString(baseIndex + 7, sendScanResult.getPermalink());
                        st.setObject(baseIndex + 8, null);
                        st.setObject(baseIndex + 9, null);
                        st.setObject(baseIndex + 10, null);
                        st.setInt(baseIndex + 11, 0);

                        baseIndex += 11;
                    }
                    return st.executeUpdate();
                });
    }

    private static class FileInfo {
        private int fileId;
        private String path;
        private String contentUrl;

        public FileInfo(int fileId, String path, String contentUrl) {
            this.fileId = fileId;
            this.path = path;
            this.contentUrl = contentUrl;
        }

        public int getFileId() {
            return fileId;
        }

        public void setFileId(int fileId) {
            this.fileId = fileId;
        }

        public String getPath() {
            return path;
        }

        public void setPath(String path) {
            this.path = path;
        }

        public String getContentUrl() {
            return contentUrl;
        }

        public void setContentUrl(String contentUrl) {
            this.contentUrl = contentUrl;
        }
    }

    private List<FileInfo> getCreativeFiles(int creativeVersionId) {
        return jdbcTemplate.query(
                "SELECT fi.nmb, fi.file_name, f.stillage_file_url" +
                        " FROM (SELECT * FROM dbo.t_creative_version WHERE nmb = ?) AS cv" +
                        " CROSS APPLY (" +
                        "   SELECT DISTINCT file_nmb" +
                        "   FROM (SELECT parameter_nmb FROM dbo.t_template_parameter WHERE template_version_nmb = cv.template_version_nmb AND dont_send_to_automoderation = 0) AS tp" +
                        "   CROSS APPLY (SELECT file_nmb FROM dbo.t_creative_version_param_value WHERE creative_version_nmb = cv.nmb AND parameter_nmb = tp.parameter_nmb) AS cvpv" +
                        " ) AS cvf" +
                        " JOIN dbo.t_file_instance AS fi ON fi.nmb = cvf.file_nmb" +
                        " JOIN dbo.t_file AS f ON f.nmb = fi.file_nmb",
                (rs, rowNum) -> new FileInfo(rs.getInt(1), rs.getString(2), rs.getString(3)),
                creativeVersionId);
    }

    private static String extractFileName(@NotNull String filePath) {
        int i = filePath.lastIndexOf("/");
        if (i == -1) {
            return filePath;
        }
        return i + 1 < filePath.length() ? filePath.substring(i + 1) : "";
    }

    private String toJson(Object o) {
        try {
            return objectMapper.writeValueAsString(o);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }
}
