package ru.yandex.bannerstorage.harvester.queues.automoderation.services;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.io.BaseEncoding;
import com.google.common.io.ByteStreams;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.Check;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.CheckResult;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.CheckResultStatus;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.Creative;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.CreativeFile;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.CreativeParamValue;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.TaskDefinition;
import ru.yandex.bannerstorage.harvester.utils.JsonUtils;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Предоставляет интеграцию креативов BS с сервисом автомодерации
 *
 * @author egorovmv
 */
public final class CreativeService {
    private static final int CHECK_RESULT_STATUS_ERROR = 1;
    private static final int CHECK_RESULT_STATUS_REJECTED = 2;
    private static final int CHECK_RESULT_STATUS_APPROVED = 3;

    private static final String CHECK_NAME_VIRUS_TOTAL = "virus_total";
    private static final String CHECK_NAME_MOBILE_STORE_LINKS = "mobile_store_link";

    private static final Logger logger = LoggerFactory.getLogger(CreativeService.class);

    private final JdbcTemplate jdbcTemplate;
    private final CloseableHttpClient httpClient;
    private final ObjectReader checkOptionsJsonReader;
    private final RowMapper<Check> checkRowMapper;
    private final ObjectWriter checkResultsJsonWriter;
    private final Supplier<Map<String, Integer>> checkNameToNmbSupplier;
    private final Supplier<Map<String, Map<String, List<String>>>> checkNameToReasonIdsSupplier;

    public CreativeService(
            @NotNull JdbcTemplate jdbcTemplate,
            CloseableHttpClient httpClient,
            int cacheTimeoutInSec) {
        Objects.requireNonNull(jdbcTemplate, "jdbcTemplate");
        Objects.requireNonNull(httpClient, "httpClient");
        if (cacheTimeoutInSec <= 0)
            throw new IllegalArgumentException("cacheTimeoutInSec: " + cacheTimeoutInSec);

        this.jdbcTemplate = jdbcTemplate;
        this.httpClient = httpClient;
        ObjectMapper objectMapper = new ObjectMapper();
        this.checkOptionsJsonReader = objectMapper.readerFor(
                new TypeReference<Map<String, Object>>() {
                });
        this.checkRowMapper = (rs, rowNum) -> new Check(
                rs.getString(1),
                JsonUtils.deserialize(
                        checkOptionsJsonReader,
                        rs.getString(2),
                        Collections::emptyMap));
        this.checkResultsJsonWriter = objectMapper.writerFor(
                new TypeReference<List<Map<String, Object>>>() {
                });

        checkNameToNmbSupplier = Suppliers.memoizeWithExpiration(
                this::getCheckNameToNmbMap, cacheTimeoutInSec, TimeUnit.SECONDS);
        checkNameToReasonIdsSupplier = Suppliers.memoizeWithExpiration(
                this::getCheckNameToReasonIdsMap, cacheTimeoutInSec, TimeUnit.SECONDS);
    }

    public Integer getCreativeId(@NotNull Integer creativeVersionId) {
        return jdbcTemplate.queryForObject(
                "SELECT creative_nmb FROM dbo.t_creative_version WHERE nmb = ?",
                Integer.class,
                creativeVersionId);
    }

    /**
     * Создать задание на автомодерацию для заданной версии креатива
     *
     * @param creativeVersionId Номер версии креатива
     * @param requiredChecks    Номера требуемых проверок (из таблицы dbo.t_automoderation_check)
     * @return Задание для автомодерации
     */
    @NotNull
    public TaskDefinition createTaskFor(@NotNull Integer creativeVersionId, @NotNull Set<Integer> requiredChecks) {
        if (requiredChecks.isEmpty())
            throw new IllegalArgumentException("requiredChecks is empty");

        @Nullable String html5MainFileUrl;
        try {
            html5MainFileUrl = jdbcTemplate.queryForObject(
                    "select " +
                            "  html5_main_file_url = f.stillage_file_url" +
                            " from t_creative_version cv" +
                            "  join t_template_version tv on cv.template_version_nmb = tv.nmb" +
                            "  join t_template t on tv.template_nmb = t.nmb" +
                            "  join t_creative_version_param_value pv on pv.creative_version_nmb = cv.nmb" +
                            "  join t_parameter p on pv.parameter_nmb = p.nmb" +
                            "  join t_file_instance fi on pv.file_nmb = fi.nmb" +
                            "  join t_file f on fi.file_nmb = f.nmb" +
                            " where cv.nmb = ? " +
                            "  and t.template_type_nmb = 8" +
                            "  and p.name = 'HTML5_FILE_MAIN_HTML'",
                    String.class,
                    creativeVersionId);
        } catch (EmptyResultDataAccessException e) {
            html5MainFileUrl = null;
        }
        //
        String code;
        if (html5MainFileUrl != null) {
            try (CloseableHttpResponse response = httpClient.execute(new HttpGet(html5MainFileUrl))) {
                HttpEntity entity = response.getEntity();
                code = new String(ByteStreams.toByteArray(entity.getContent()), UTF_8);
                EntityUtils.consume(entity);
            } catch (IOException e) {
                throw new RuntimeException(
                        String.format(
                                "Can't read the code from stillage: %s (creative_version_nmb = %d).",
                                html5MainFileUrl,
                                creativeVersionId
                        ));
            }
        } else {
            code = new String(
                    jdbcTemplate.queryForObject(
                            "SELECT dbo.clr_uf_creative_version_generate_code2(?, 1, -1, 0, NULL)",
                            byte[].class,
                            creativeVersionId),
                    UTF_8);
        }

        List<CreativeFile> files = jdbcTemplate.query(
                "SELECT fi.nmb, f.hash, fi.file_name, f.stillage_file_url" +
                        " FROM (SELECT * FROM dbo.t_creative_version WHERE nmb = ?) as cv" +
                        " CROSS APPLY (" +
                        "   SELECT DISTINCT file_nmb" +
                        "   FROM (SELECT parameter_nmb FROM dbo.t_template_parameter WHERE template_version_nmb = cv.template_version_nmb AND dont_send_to_automoderation = 0) AS tp" +
                        "   CROSS APPLY (SELECT file_nmb FROM dbo.t_creative_version_param_value WHERE creative_version_nmb = cv.nmb AND parameter_nmb = tp.parameter_nmb) AS cvpv" +
                        " ) as cvf" +
                        " JOIN dbo.t_file_instance AS fi ON fi.nmb = cvf.file_nmb" +
                        " JOIN dbo.t_file AS f ON f.nmb = fi.file_nmb",
                (rs, rowNum) -> {
                    return new CreativeFile(
                            Integer.toString(rs.getInt(1)),
                            BaseEncoding.base16().encode(rs.getBytes(2)),
                            rs.getString(3),
                            rs.getString(4));
                },
                creativeVersionId);


        List<CreativeParamValue> paramValues = jdbcTemplate.query(
                "DECLARE @nmbs type_id_set; " +
                        "INSERT INTO @nmbs VALUES(?); " +
                        "SELECT param_name, param_value, order_in_list, param_type  FROM dbo.uf_creative_version_get_param_values(@nmbs, 0, 0, 0, 0, 1); ",
                (rs, rowNum) -> new CreativeParamValue(
                        rs.getString(1),
                        rs.getString(2),
                        (Integer) rs.getObject(3),
                        rs.getString(4)
                ),
                creativeVersionId);

        if (html5MainFileUrl != null) {
            Optional<CreativeFile> mainHtmlFileOptional = files.stream().filter(f -> {
                String lowercasedPath = f.getPath().toLowerCase();
                return lowercasedPath.endsWith(".html") || lowercasedPath.endsWith(".htm");
            }).findAny();
            if (!mainHtmlFileOptional.isPresent()) {
                throw new AssertionError("This shouldn't happen");
            }
            CreativeFile mainHtmlFile = mainHtmlFileOptional.get();
            CreativeFile indexHtmlFile = new CreativeFile(
                    mainHtmlFile.getId(),
                    mainHtmlFile.getMd5(),
                    "index.html",
                    mainHtmlFile.getContentUrl()
            );
            files.remove(mainHtmlFile);
            files.add(indexHtmlFile);
        }

        List<Check> checks = jdbcTemplate.query(
                "SELECT ch.name, tvch.options_json" +
                        " FROM (SELECT * FROM dbo.t_creative_version WHERE nmb = ?) as cv" +
                        " JOIN dbo.t_template_version_automoderation_check AS tvch ON tvch.template_version_nmb = cv.template_version_nmb" +
                        " JOIN dbo.t_automoderation_check AS ch ON ch.nmb = tvch.automoderation_check_nmb" +
                        " WHERE tvch.is_enabled = 1 AND tvch.automoderation_check_nmb IN (SELECT id FROM dbo.uf_SplitWithCTE(?, ',', 1))" +
                        // проверки Phantom JS и MOBILE_STORE_LINK обрабатываем отдельно
                        "  AND tvch.automoderation_check_nmb NOT IN (1, 2, 4)" +
                        " OPTION (maxrecursion 0)",
                checkRowMapper,
                creativeVersionId,
                requiredChecks.stream()
                        .map(Object::toString)
                        .collect(Collectors.joining(",")));

        return new TaskDefinition(new Creative(
                html5MainFileUrl != null ? null : code, files, paramValues), checks);
    }

    /**
     * Удалить старые результаты автомодерации для данной версии креатива
     *
     * @param creativeVersionId Номер версии креатива
     * @param checks    Номера требуемых проверок (из таблицы dbo.t_automoderation_check)
     */
    public void removeOldResults(@NotNull Integer creativeVersionId, @NotNull Set<Integer> checks) {
        if (checks.isEmpty())
            throw new IllegalArgumentException("requiredChecks is empty");

        jdbcTemplate.update(
                "DELETE FROM cvr" +
                        " FROM dbo.r_creative_version_reason AS cvr" +
                        " JOIN dbo.t_creative_version AS cv ON cv.nmb = cvr.creative_version_nmb" +
                        " JOIN dbo.t_template_version_automoderation_check AS tvch ON tvch.template_version_nmb = cv.template_version_nmb" +
                        " WHERE cvr.creative_version_nmb = ?" +
                        "   AND tvch.automoderation_check_nmb IN (SELECT id FROM dbo.uf_SplitWithCTE(?, ',', 1))" +
                        "   AND cvr.is_by_automoderator = 1",
                creativeVersionId,
                checks.stream()
                        .map(Object::toString)
                        .collect(Collectors.joining(",")));
    }

    private Map<String, Integer> getCheckNameToNmbMap() {
        return jdbcTemplate.queryForList("SELECT name, nmb FROM dbo.t_automoderation_check")
                .stream()
                .collect(Collectors.toMap(r -> (String) r.get("name"), r -> (Integer) r.get("nmb")));
    }

    /**
     *
     * @return checkname -> reject_alias -> reject_reason_nmb
     */
    private Map<String, Map<String, List<String>>> getCheckNameToReasonIdsMap() {
        return jdbcTemplate.queryForList(
                "SELECT ch.name, r.id, chr.reject_reason_alias" +
                        " FROM dbo.t_automoderation_check AS ch" +
                        " JOIN dbo.t_automoderation_check_reject_reason AS chr ON chr.automoderation_check_nmb = ch.nmb" +
                        " JOIN dbo.d_reason AS r ON r.nmb = chr.reason_nmb")
                .stream()
                .collect(
                        Collectors.groupingBy(
                                r -> (String) r.get("name"),
                                Collectors.groupingBy(r -> (String) r.get("reject_reason_alias"),
                                        Collectors.mapping(r -> (String) r.get("id"), Collectors.toList()))));
    }

    /**
     * Сохранить новые результаты проверок для данной версии креатива
     *
     * @param creativeVersionId Номер версии креатива
     * @param newCheckResults   Новые результаты проверок
     */
    public void append(@NotNull Integer creativeVersionId, List<CheckResult> newCheckResults) {
        List<Object[]> checkResultBatch = new ArrayList<>();
        List<Object[]> rejectReasonBatch = new ArrayList<>();

        Map<String, Integer> checkNameToNmb = null;
        Map<String, Map<String, List<String>>> checkNameToReasonIds = null;

        for (CheckResult checkResult : newCheckResults) {
            String resultsAsJson = JsonUtils.serialize(
                    checkResultsJsonWriter, checkResult.getResults());
            if (checkResult.getStatus().isError()) {
                checkResultBatch.add(
                        new Object[]{
                                creativeVersionId,
                                checkResult.getName(),
                                resultsAsJson,
                                CHECK_RESULT_STATUS_ERROR});
                continue;
            }
            if (checkNameToNmb == null)
                checkNameToNmb = checkNameToNmbSupplier.get();

            checkResultBatch.add(
                    new Object[]{
                            creativeVersionId,
                            checkNameToNmb.get(checkResult.getName()),
                            resultsAsJson,
                            checkResult.getStatus() != CheckResultStatus.PASSED ? CHECK_RESULT_STATUS_REJECTED
                                    : CHECK_RESULT_STATUS_APPROVED});

            if (checkResult.getStatus() == CheckResultStatus.FAILED) {
                if (checkNameToReasonIds == null)
                    checkNameToReasonIds = checkNameToReasonIdsSupplier.get();

                List<String> reasonIds = checkNameToReasonIds.getOrDefault(
                        checkResult.getName(), Collections.emptyMap()).values().stream().flatMap(Collection::stream).collect(Collectors.toList());

                if (CHECK_NAME_VIRUS_TOTAL.equalsIgnoreCase(checkResult.getName())) {
                    for (String reasonId : reasonIds) {
                        for (Map<String, Object> infectedFiles : checkResult.getResults()) {
                            rejectReasonBatch.add(
                                    new Object[]{
                                            creativeVersionId,
                                            reasonId,
                                            Integer.valueOf((String) infectedFiles.get("id"))});
                        }
                    }
                    continue;
                }

                if (CHECK_NAME_MOBILE_STORE_LINKS.equalsIgnoreCase(checkResult.getName())) {
                    for (Map<String, Object> failParams : checkResult.getResults()) {
                        String code = (String) failParams.get("reject_reason_alias");
                        if (code == null) {
                            logger.error("Got " + CHECK_NAME_MOBILE_STORE_LINKS + " check results for creative version " + creativeVersionId + " but it has no required result param \"code\"" );
                            continue;
                        }
                        List<String> rejectReasonIds = checkNameToReasonIds.get(CHECK_NAME_MOBILE_STORE_LINKS).get(code);
                        if (rejectReasonIds.isEmpty()) {
                            logger.error("Got no applicable reject reason id for reject reason alias " + code + " and creative version " + creativeVersionId);
                            continue;
                        }
                        rejectReasonBatch.add(
                                new Object[]{
                                        creativeVersionId,
                                        rejectReasonIds.get(0),
                                        null});
                    }
                    continue;
                }

                for (String reasonId : reasonIds) {
                    rejectReasonBatch.add(new Object[]{creativeVersionId, reasonId, null});
                }
            }
        }

        jdbcTemplate.batchUpdate(
                "DECLARE @creativeVersionNmb INT = ?;" +
                        " DECLARE @checkNmb INT = ?;" +
                        " INSERT dbo.t_creative_version_check_result (creative_version_nmb, check_nmb, results_json, status_nmb)" +
                        " SELECT @creativeVersionNmb, tvch.nmb, ?, ?" +
                        " FROM dbo.t_template_version_automoderation_check as tvch" +
                        " WHERE template_version_nmb = (" +
                        "   SELECT template_version_nmb FROM dbo.t_creative_version WHERE nmb = @creativeVersionNmb" +
                        " ) AND automoderation_check_nmb = @checkNmb",
                checkResultBatch);

        if (!rejectReasonBatch.isEmpty()) {
            jdbcTemplate.batchUpdate(
                    "MERGE dbo.r_creative_version_reason AS t" +
                            " USING (VALUES(?, ?, ?)) as s(creative_version_nmb, reason_id, macros_nmb)" +
                            " ON t.creative_version_nmb = s.creative_version_nmb" +
                            "   AND t.reason_id = s.reason_id" +
                            "   AND t.macros_nmb = s.macros_nmb" +
                            " WHEN NOT MATCHED THEN" +
                            " INSERT (creative_version_nmb, reason_id, macros_nmb, is_by_automoderator)" +
                            " VALUES (s.creative_version_nmb, s.reason_id, s.macros_nmb, 1);",
                    rejectReasonBatch);
        }
    }



    public boolean isReadyForModeration(@NotNull Integer creativeVersionId) {
        int incompletedCount = jdbcTemplate.queryForObject(
                "SELECT COUNT(*)" +
                        " FROM (SELECT * FROM dbo.t_creative_version WHERE nmb = ?) AS cv" +
                        " JOIN (" +
                        "   SELECT *" +
                        "   FROM dbo.t_template_version_automoderation_check" +
                        "   WHERE is_required = 1 AND is_enabled = 1" +
                        " ) AS tvch ON tvch.template_version_nmb = cv.template_version_nmb" +
                        " WHERE NOT EXISTS(" +
                        "   SELECT *" +
                        "   FROM dbo.t_creative_version_check_result" +
                        "   WHERE creative_version_nmb = cv.nmb" +
                        "       AND check_nmb = tvch.nmb" +
                        "       AND status_nmb IN (3, 4)" + // approved, approved_with_warnings
                        " )",
                Integer.class,
                creativeVersionId);
        return incompletedCount == 0;
    }

    public boolean hasNewModeratedVersion(@NotNull Integer creativeVersionId) {
        return jdbcTemplate.queryForObject(
                "DECLARE @creativeVersionNmb INT = ?;" +
                        " SELECT " +
                        " CASE WHEN EXISTS(" +
                        "   SELECT *" +
                        "   FROM dbo.t_creative_version" +
                        "   WHERE creative_nmb = (SELECT creative_nmb FROM dbo.t_creative_version WHERE nmb = @creativeVersionNmb)" +
                        "       AND nmb > @creativeVersionNmb" +
                        "       AND status_nmb IN (3, 4)) " +
                        " THEN 1" +
                        " ELSE 0" +
                        " END",
                Boolean.class,
                creativeVersionId);
    }

    /**
     * Отклонить данную версию креатива
     *
     * @param creativeVersionId Номер версии креатива
     * @return true, если данная версия креатива была уже принята модератором, иначе false
     */
    public boolean reject(Integer creativeVersionId) {
        List<Integer> oldStatusNmb = jdbcTemplate.queryForList(
                "DECLARE @creativeVersionNmb INT = ?;" +
                        " UPDATE dbo.t_creative_version" +
                        " SET status_nmb = 3," +
                        "   is_enabled = 0," +
                        "   moderation_date = GETDATE()," +
                        "   is_rejected_by_automoderator = 1" +
                        " OUTPUT deleted.status_nmb" +
                        " WHERE nmb = @creativeVersionNmb AND status_nmb in (2, 4)",
                Integer.class,
                creativeVersionId);
        return !oldStatusNmb.isEmpty() && oldStatusNmb.get(0) == 4;
    }

    /**
     * Пометить данную версию креатива, как готовую к модерации
     *
     * @param creativeVersionId Номер версии креатива
     */
    public void moveToModeration(@NotNull Integer creativeVersionId) {
        jdbcTemplate.update(
                "UPDATE dbo.t_creative_version" +
                        " SET is_ready_to_moderation = 1, " +
                        "   automoderation_request_date = GETDATE()" +
                        " WHERE nmb = ? AND status_nmb = 2",
                creativeVersionId);
    }

    /**
     * Проверить отправлялась ли одна из предыдущих версий креатива в БК.
     * Нужно, чтобы понять отправлять ли в БК уведомление о отклонении текущей версии креатива
     *
     * @param creativeVersionId Номер версии креатива
     * @return true, если отправлялась, иначе false
     */
    public boolean hasAnyPrevVersionSentToBk(@NotNull Integer creativeVersionId) {
        return jdbcTemplate.queryForObject(
                "DECLARE @creativeVersionNmb INT = ?" +
                        " SELECT " +
                        " CASE WHEN EXISTS(" +
                        "   SELECT *" +
                        "   FROM dbo.t_creative_version" +
                        "   WHERE creative_nmb = (SELECT creative_nmb FROM dbo.t_creative_version WHERE nmb = @creativeVersionNmb)" +
                        "       AND nmb < @creativeVersionNmb" +
                        "       AND status_nmb = 4) " +
                        " THEN 1" +
                        " ELSE 0" +
                        " END",
                Boolean.class,
                creativeVersionId);
    }

    /**
     * Послать уведомление в БК о изменениях в креативе
     *
     * @param creativeVersionId Номер версии креатива
     */
    public void notifyBk(@NotNull Integer creativeVersionId) {
        jdbcTemplate.update(
                "EXEC dbo.sp_rtbhostlink_notify_creative_changed ?",
                creativeVersionId);
    }

    /**
     * Послать уведомление Customer-у о изменении в креативе
     *
     * @param creativeVersionId Номер версии креатива
     */
    public void notifyCustomer(@NotNull Integer creativeVersionId) {
        jdbcTemplate.update(
                "DECLARE @creativeVersionNmb INT = ?;" +
                        " UPDATE dbo.t_creative_version " +
                        " SET notified = 0" +
                        " WHERE creative_nmb = (SELECT creative_nmb FROM dbo.t_creative_version WHERE nmb = @creativeVersionNmb)" +
                        "   AND nmb >= @creativeVersionNmb;" +
                        " EXEC dbo.sp_notify_creative_customer_service_notify_creative_changed @creativeVersionNmb",
                creativeVersionId);

    }
}
