package ru.yandex.bannerstorage.harvester.queues.automoderation.services;

import java.util.List;
import java.util.Objects;

import org.jetbrains.annotations.NotNull;

import ru.yandex.bannerstorage.harvester.infrastructure.AsyncEmailMessageSender;
import ru.yandex.bannerstorage.harvester.queues.automoderation.AutoModerationNotifyQueueObserver;
import ru.yandex.bannerstorage.harvester.queues.automoderation.models.AutoModerationNotifyRequest;
import ru.yandex.bannerstorage.harvester.queues.automoderation.services.models.CheckResult;
import ru.yandex.bannerstorage.messaging.services.QueueMessageSender;

/**
 * @author egorovmv
 */
public final class CreativeWorkflowService {
    private final CreativeService creativeService;
    private final QueueMessageSender queueMessageSender;
    private final AsyncEmailMessageSender emailMessageSender;
    private final String[] rejectRecipients;
    private final String premoderationUrl;

    public CreativeWorkflowService(
            @NotNull CreativeService creativeService,
            @NotNull QueueMessageSender queueMessageSender,
            @NotNull AsyncEmailMessageSender emailMessageSender,
            @NotNull String[] rejectRecipients,
            @NotNull String premoderationUrl) {
        Objects.requireNonNull(creativeService, "creativeService");
        Objects.requireNonNull(queueMessageSender, "queueMessageSender");
        Objects.requireNonNull(emailMessageSender, "emailMessageSender");
        this.creativeService = creativeService;
        this.queueMessageSender = queueMessageSender;
        this.emailMessageSender = emailMessageSender;
        this.rejectRecipients = rejectRecipients;
        this.premoderationUrl = premoderationUrl;
    }

    private static String buildRejectMessageSubject(@NotNull Integer creativeId) {
        return String.format("Креатив \"%s\" был отклонен автомодерацией", creativeId);
    }

    private String buildEmailMessage(
            @NotNull Integer creativeVersionId,
            @NotNull List<CheckResult> failedCheckResults,
            boolean rejecting) {
        StringBuilder builder = new StringBuilder();
        if (rejecting) {
            builder.append("Креатив отклонён:\n");
        } else {
            builder.append("Внимание, были найдены проблемы в креативе:\n");
        }
        builder.append("Ссылка:\n");
        builder.append(premoderationUrl).append("/creative/index.jsp?nmb=")
                .append(creativeVersionId);
        builder.append("\n\nПричины:\n");
        for (CheckResult checkResult : failedCheckResults) {
            builder.append(checkResult.getName())
                    .append(":")
                    .append(checkResult.getResults().toString());
        }
        if (!rejecting) {
            builder.append("\nКреатив НЕ БУДЕТ отклонён автоматически");
            builder.append("\nЕсли креатив нужно отклонить, вы можете это сделать в интерфейсе модерации");
        }
        return builder.toString();
    }

    public void reject(
            @NotNull Integer creativeVersionId, @NotNull List<CheckResult> failedCheckResults) {
        if (!creativeService.hasNewModeratedVersion(creativeVersionId)) {
            boolean wasApprovedByModerator = creativeService.reject(creativeVersionId);

            Integer creativeId = creativeService.getCreativeId(creativeVersionId);

            queueMessageSender.sendMessage(
                    AutoModerationNotifyQueueObserver.SERVICE_ID,
                    new AutoModerationNotifyRequest(creativeVersionId, wasApprovedByModerator));

            emailMessageSender.sendMessage(
                    rejectRecipients,
                    buildRejectMessageSubject(creativeId),
                    buildEmailMessage(creativeVersionId, failedCheckResults, true));
        }
    }
}
