package ru.yandex.bannerstorage.harvester.queues.processdynamiccode;

import java.io.InputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Objects;

import javax.inject.Inject;

import org.jetbrains.annotations.NotNull;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.exceptions.CantParseHtmlDocumentException;
import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.exceptions.InvalidCodeFileException;
import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.exceptions.InvalidUriException;
import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.infrastracture.FileStorageService;
import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.infrastracture.ImageStorageService;
import ru.yandex.bannerstorage.messaging.services.ReplyQueueMessage;
import ru.yandex.bannerstorage.messaging.services.SimpleRequestReplyQueueObserver;

/**
 * @author egorovmv
 */
public final class ProcessDynamicCodeQueueObserver extends SimpleRequestReplyQueueObserver<ProcessDynamicCodeRequest, ProcessDynamicCodeResponse> {
    private static final Logger logger = LoggerFactory.getLogger(ProcessDynamicCodeQueueObserver.class);

    private static final String QUEUE_ID = "dbo.CreativeProcessDynamicCodeServiceQueue";

    private static final String RESPONSE_MESSAGE_TYPE = "https://bannerstorage.yandex-team.ru/creative/process-dynamic-code/response-message";

    private final FileStorageService fileStorageService;
    private final ImageStorageService imageStorageService;
    private final HtmlCodeFileProcessor htmlCodeFileProcessor;

    @Inject
    public ProcessDynamicCodeQueueObserver(
            @NotNull FileStorageService fileStorageService,
            @NotNull ImageStorageService imageStorageService) {
        super(
                QUEUE_ID,
                DEFAULT_POLL_INTERVAL_IN_MS,
                ProcessDynamicCodeRequest.class);
        this.fileStorageService = Objects.requireNonNull(fileStorageService, "fileStorageService");
        this.imageStorageService = Objects.requireNonNull(imageStorageService, "imageStorageService");
        this.htmlCodeFileProcessor = new HtmlCodeFileProcessor();
    }

    private static ProcessDynamicCodeResponse createErrorResponse(
            @NotNull Integer creativeVersionId, int creativeVersionSubVersionId, InvalidCodeFileException cause) {
        String rejectCode;
        String rejectDetails;
        if (cause instanceof InvalidUriException) {
            rejectCode = "InvalidUri";
            rejectDetails = cause.getMessage();
        } else {
            rejectCode = "InvalidFile";
            rejectDetails = "";
        }
        return ProcessDynamicCodeResponse.error(
                creativeVersionId, creativeVersionSubVersionId, rejectCode, rejectDetails);
    }

    private Document getAndParseCodeFileAsHtmlDocument(Integer creativeVersionId) {
        try (InputStream fileContent = new URL(fileStorageService.getFileParameterContentUrl(creativeVersionId, "CODE_FILE")).openStream()) {
            return Jsoup.parse(
                    fileContent,
                    StandardCharsets.UTF_8.name(),
                    "");
        } catch (Exception e) {
            logger.error("getAndParseCodeFileAsHtmlDocument error", e);
            throw new CantParseHtmlDocumentException(e);
        }
    }

    @Override
    public ReplyQueueMessage<ProcessDynamicCodeResponse> doProcessMessage(
            @NotNull ProcessDynamicCodeRequest request) {
        Integer creativeVersionId = request.getCreativeVersionId();
        try (ImageStorageService.ImageUploader imageUploader = imageStorageService.createUploader()) {
            try {
                Document htmlCodeFile = getAndParseCodeFileAsHtmlDocument(creativeVersionId);

                // Преобразуем html-файл подставляя вместо старых ссылок на картинки ссылки на аватараницу
                htmlCodeFileProcessor.processDocument(htmlCodeFile, imageUploader);

                // Сохраняем обработанный файл с новыми ссылками
                htmlCodeFile.outputSettings().prettyPrint(false);
                Integer processCodeFileId = fileStorageService.saveFileWithCustomerByCreativeVersion(
                        creativeVersionId,
                        htmlCodeFile.toString().getBytes(StandardCharsets.UTF_8));

                // Считаем, что теперь загруженные в Аватарницу картинки удалять точно не нужно.
                // Правда есть риск что транзакция не закомитится, но по идеи он минимален.
                // Худшее что в таком случае нам грозит это дублирование картинок а Аватарнице
                imageUploader.commit();

                // Отправляем назад уведомление, что нам удалось обработать CODE_FILE
                return new ReplyQueueMessage<>(
                        RESPONSE_MESSAGE_TYPE,
                        ProcessDynamicCodeResponse.success(
                                creativeVersionId, request.getCreativeVersionSubVersionId(), processCodeFileId));
            } catch (InvalidCodeFileException e) {
                // Отправляем назад уведомление, что данную версию креатива надо отклонять
                return new ReplyQueueMessage<>(
                        RESPONSE_MESSAGE_TYPE,
                        createErrorResponse(creativeVersionId, request.getCreativeVersionSubVersionId(), e));
            }
        }
    }
}
