package ru.yandex.bannerstorage.harvester.queues.processdynamiccode.infrastracture.impl;

import java.util.Arrays;

import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.exceptions.InvalidUriException;
import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.infrastracture.ImageStorageService;

/**
 * @author egorovmv
 */
public abstract class AbstractImageStorageService implements ImageStorageService {
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractImageStorageService.class);

    @Override
    public abstract ImageUploader createUploader();

    protected static abstract class AbstractImageUploader implements ImageUploader {
        private static final String[] PROTOCOL_PREFIXES = {"http:", "https:"};

        private static String makeAbsoluteUrl(String protocolPrefix, String relativeUrl) {
            if (!relativeUrl.startsWith("//"))
                return protocolPrefix + "//" + relativeUrl;
            else
                return protocolPrefix + relativeUrl;
        }

        @NotNull
        protected abstract String doUploadImage(@NotNull String originalImageUrl);

        @NotNull
        @Override
        public final String uploadImage(@NotNull String originalImageUrl) {
            if (Arrays.stream(PROTOCOL_PREFIXES).anyMatch(originalImageUrl::startsWith))
                return doUploadImage(originalImageUrl);
            for (String protocolPrefix : PROTOCOL_PREFIXES) {
                String possibleAbsoluteImageUrl = makeAbsoluteUrl(protocolPrefix, originalImageUrl);
                try {
                    String newImageUrl = doUploadImage(possibleAbsoluteImageUrl);
                    int startIndex = protocolPrefix.length();
                    if (!originalImageUrl.startsWith("//"))
                        startIndex += 2;
                    return newImageUrl.substring(startIndex);
                } catch (InvalidUriException e) {
                    // У нас не получилось загрузить с заданным префиксом протокола, игнорим и пробуем другой
                    LOGGER.error(String.format("Invalid image url \"%s\"", possibleAbsoluteImageUrl), e);
                }
            }
            // Мы исчерпали все попытки, считаем картинку плохой
            throw new InvalidUriException(originalImageUrl);
        }

        @Override
        public void commit() {
        }

        @Override
        public void close() {
        }
    }
}
