package ru.yandex.bannerstorage.harvester.queues.processdynamiccode.infrastracture.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

import javax.inject.Inject;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpServerErrorException;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.client.UnknownHttpStatusCodeException;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.exceptions.ImageStorageFailedException;
import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.exceptions.InvalidUriException;
import ru.yandex.bannerstorage.harvester.utils.RestTemplateFactory;
import ru.yandex.bannerstorage.messaging.services.exceptions.AbortMessageProcessingException;

/**
 * @author egorovmv
 */
public final class AvatarnicaImageStorageService extends AbstractImageStorageService {
    private static final Logger LOGGER = LoggerFactory.getLogger(AvatarnicaImageStorageService.class);

    private static final int DEFAULT_TIMEOUT_IN_MS = 50000;

    private static final int AVATARNICA_INVALID_IMAGE_URL_HTTP_STATUS_CODE = 434;

    private final RestTemplate restTemplate;
    private final String readUrl;
    private final String writeUrl;
    private final String namespace;

    @Inject
    public AvatarnicaImageStorageService(
            @NotNull String readUrl, @NotNull String writeUrl, @NotNull String namespace) {
        LOGGER.info(
                "Constructing (readUrl: \"{}\", writeUrl: \"{}\", namespace: \"{}\")...",
                readUrl, writeUrl, namespace);

        this.restTemplate = RestTemplateFactory.newInstance(DEFAULT_TIMEOUT_IN_MS);
        this.readUrl = Objects.requireNonNull(readUrl, "readUrl");
        this.writeUrl = Objects.requireNonNull(writeUrl, "writeUrl");
        this.namespace = Objects.requireNonNull(namespace, "namespace");

        LOGGER.info("Constructed");
    }

    @Override
    public ImageUploader createUploader() {
        return new AvatarnicaImageUploader();
    }

    private static class UploadImageResponse {
        @JsonProperty("group-id")
        private String groupId;
        @JsonProperty
        private ImageMeta meta;
        @JsonProperty("sizes")
        private ImageSizeCollection sizes;

        @Override
        public String toString() {
            return "UploadImageResponse{" +
                    "groupId='" + groupId + '\'' +
                    ", sizes=" + sizes +
                    '}';
        }

        public String getGroupId() {
            return groupId;
        }

        public void setGroupId(String groupId) {
            this.groupId = groupId;
        }

        public ImageMeta getMeta() {
            return meta;
        }

        public void setMeta(ImageMeta meta) {
            this.meta = meta;
        }

        public ImageSizeCollection getSizes() {
            return sizes;
        }

        public void setSizes(ImageSizeCollection sizes) {
            this.sizes = sizes;
        }

        private static class ImageMeta {
            @JsonProperty
            private String md5;
            @JsonProperty("orig-format")
            private String orignalFormat;
            @JsonProperty("orig-size")
            private OriginalSize originalSize;
            @JsonProperty("processed_by_computer_vision")
            private String processedByComputerVision;

            @Override
            public String toString() {
                return "ImageMeta{" +
                        "md5='" + md5 + '\'' +
                        ", orignalFormat='" + orignalFormat + '\'' +
                        ", originalSize=" + originalSize +
                        ", processedByComputerVision='" + processedByComputerVision + '\'' +
                        '}';
            }

            public String getMd5() {
                return md5;
            }

            public void setMd5(String md5) {
                this.md5 = md5;
            }

            public String getOrignalFormat() {
                return orignalFormat;
            }

            public void setOrignalFormat(String orignalFormat) {
                this.orignalFormat = orignalFormat;
            }

            public OriginalSize getOriginalSize() {
                return originalSize;
            }

            public void setOriginalSize(OriginalSize originalSize) {
                this.originalSize = originalSize;
            }

            public String getProcessedByComputerVision() {
                return processedByComputerVision;
            }

            public void setProcessedByComputerVision(String processedByComputerVision) {
                this.processedByComputerVision = processedByComputerVision;
            }

            private static class OriginalSize {
                @JsonProperty
                private int x;
                @JsonProperty
                private int y;

                @Override
                public String toString() {
                    return "OriginalSize{" +
                            "x=" + x +
                            ", y=" + y +
                            '}';
                }

                public int getX() {
                    return x;
                }

                public void setX(int x) {
                    this.x = x;
                }

                public int getY() {
                    return y;
                }

                public void setY(int y) {
                    this.y = y;
                }
            }
        }

        private static class ImageSize {
            @JsonProperty
            private String path;
            @JsonProperty
            private int height;
            @JsonProperty
            private int width;

            @Override
            public String toString() {
                return "ImageSize{" +
                        "path='" + path + '\'' +
                        ", height=" + height +
                        ", width=" + width +
                        '}';
            }

            public String getPath() {
                return path;
            }

            public void setPath(String path) {
                this.path = path;
            }

            public int getHeight() {
                return height;
            }

            public void setHeight(int height) {
                this.height = height;
            }

            public int getWidth() {
                return width;
            }

            public void setWidth(int width) {
                this.width = width;
            }
        }

        public static class ImageSizeCollection {
            @JsonProperty("orig")
            private ImageSize original;

            @Override
            public String toString() {
                return "ImageSizeCollection{" +
                        "original=" + original +
                        '}';
            }

            public ImageSize getOriginal() {
                return original;
            }

            public void setOriginal(ImageSize original) {
                this.original = original;
            }
        }
    }

    private class AvatarnicaImageUploader extends AbstractImageUploader {
        private final String readCommand;
        private final String uploadCommand;
        private final String deleteCommand;
        private final List<String> uploadedImagePaths;

        private AvatarnicaImageUploader() {
            this.readCommand = "get-" + namespace;
            this.uploadCommand = "put-" + namespace;
            this.deleteCommand = "delete-" + namespace;
            this.uploadedImagePaths = new ArrayList<>();
        }

        private UploadImageResponse doExecuteRequest(URI requestUri) {
            HttpHeaders headers = new HttpHeaders();
            headers.setAccept(
                    Collections.singletonList(
                            MediaType.valueOf(MediaType.APPLICATION_JSON_VALUE)));

            ResponseEntity<UploadImageResponse> response = restTemplate.exchange(
                    requestUri,
                    HttpMethod.GET,
                    new HttpEntity<>(headers),
                    UploadImageResponse.class);

            return response.getBody();
        }

        private String executeRequest(@NotNull String originalImageUrl) {
            try {
                String imageId = UUID.randomUUID().toString();

                URI requestUri = UriComponentsBuilder.fromHttpUrl(writeUrl)
                        .pathSegment(uploadCommand, imageId)
                        .queryParam("url", originalImageUrl)
                        .build()
                        .toUri();

                LOGGER.info("Executing request \"{}\"", requestUri);
                UploadImageResponse response = doExecuteRequest(requestUri);
                LOGGER.info("Request \"{}\" executed (Response: {})", requestUri, response);

                return response.getGroupId() + '/' + imageId;
            } catch (ResourceAccessException | HttpServerErrorException e) {
                // Если сломались из-за того что произошли какие-то сетевые проблемы, то надо попробовать обработать
                // текущее сообщение еще раз через некоторый промежуток времени (вдруг все починится). Поэтому
                // перепланируем обработку текущего сообщения
                throw new AbortMessageProcessingException(e);
            } catch (HttpClientErrorException e) {
                // Если делаем слишком много запросов, то перепланируем обработку через какой-то промежуток времени
                // иначе скорее всего что-то не так с нашей картинкой
                if (e.getStatusCode() == HttpStatus.TOO_MANY_REQUESTS)
                    throw new AbortMessageProcessingException(e);
                else
                    throw new InvalidUriException(originalImageUrl, e);
            } catch (UnknownHttpStatusCodeException e) {
                // Аватарница не смогла загрузить нашу картинку (скорее всего с ней что-то не так)
                // иначе считаем, что проблемы у Аватарницы и перезапускаем
                if (e.getRawStatusCode() == AVATARNICA_INVALID_IMAGE_URL_HTTP_STATUS_CODE)
                    throw new InvalidUriException(originalImageUrl, e);
                else
                    throw new ImageStorageFailedException(originalImageUrl, e);
            }
        }

        @NotNull
        @Override
        public String doUploadImage(@NotNull String originalImageUrl) {
            LOGGER.info("Uploading image \"{}\"...", originalImageUrl);
            String imagePath = executeRequest(originalImageUrl);
            uploadedImagePaths.add(imagePath);
            String result = UriComponentsBuilder.fromHttpUrl(readUrl)
                    .pathSegment(readCommand, imagePath, "orig")
                    .build()
                    .toString();
            LOGGER.info("Image \"{}\" uploaded (Result: \"{}\")", originalImageUrl, result);
            return result;
        }

        @Override
        public void commit() {
            uploadedImagePaths.clear();
        }

        private boolean removeImage(@NotNull String imagePath) {
            try {
                final URI requestUri = UriComponentsBuilder.fromHttpUrl(writeUrl)
                        .pathSegment(deleteCommand, imagePath)
                        .build()
                        .toUri();

                HttpHeaders headers = new HttpHeaders();
                headers.setAccept(
                        Collections.singletonList(
                                MediaType.valueOf(MediaType.APPLICATION_JSON_VALUE)));

                restTemplate.exchange(
                        requestUri,
                        HttpMethod.POST,
                        new HttpEntity<>(headers),
                        String.class);

                return true;
            } catch (Exception e) {
                LOGGER.error("Can't remove image \"{}\"", imagePath);
                return false;
            }
        }

        @Override
        public void close() {
            uploadedImagePaths.forEach(this::removeImage);
        }
    }
}
