package ru.yandex.bannerstorage.harvester.queues.processdynamiccode.infrastracture.impl;

import java.sql.Types;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.inject.Inject;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.simple.SimpleJdbcInsert;

import ru.yandex.bannerstorage.harvester.integration.stillage.StillageAnswer;
import ru.yandex.bannerstorage.harvester.integration.stillage.StillageClient;
import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.exceptions.FileParameterNotFoundException;
import ru.yandex.bannerstorage.harvester.queues.processdynamiccode.infrastracture.FileStorageService;

/**
 * @author egorovmv
 */
public final class FileStorageServiceImpl implements FileStorageService {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileStorageServiceImpl.class);

    private final JdbcTemplate jdbcTemplate;
    private final StillageClient stillageClient;

    @Inject
    public FileStorageServiceImpl(@NotNull JdbcTemplate jdbcTemplate,
                                  @NotNull StillageClient stillageClient) {
        this.jdbcTemplate = Objects.requireNonNull(jdbcTemplate, "jdbcTemplate");
        this.stillageClient = Objects.requireNonNull(stillageClient, "stillageClient");
    }

    @NotNull
    @Override
    public String getFileParameterContentUrl(
            @NotNull Integer creativeVersionId, @NotNull String parameterName) {
        LOGGER.info("Reading file parameter content (CreativeVersionId: {})...", creativeVersionId);

        final String fileContentUrl = jdbcTemplate.queryForObject(
                "select stillage_file_url\n" +
                        "from dbo.t_file with(nolock)\n" +
                        "where nmb = (" +
                        "select file_nmb\n" +
                        "from dbo.t_file_instance with(nolock)\n" +
                        "where nmb = (" +
                        "select file_nmb\n" +
                        "from dbo.t_creative_version_param_value with(nolock)\n" +
                        "where creative_version_nmb = ? and\n" +
                        "parameter_nmb = (select nmb from dbo.t_parameter with(nolock) where [name] = ?)))",
                String.class,
                creativeVersionId,
                parameterName);

        if (fileContentUrl == null)
            throw new FileParameterNotFoundException(creativeVersionId, parameterName);

        LOGGER.info("File parameter content read CreativeVersionId: {}, fileContentUrl: {}", creativeVersionId,
                fileContentUrl);

        return fileContentUrl;
    }

    @Override
    public Integer saveFileWithCustomerByCreativeVersion(
            @NotNull Integer creativeVersionId, @NotNull byte[] fileContent) {
        LOGGER.info("Saving file content (CreativeVersionId: {})...", creativeVersionId);

        final StillageAnswer stillageAnswer = stillageClient.uploadFile("processed_code_file.html", fileContent);

        final Integer customerNmb = jdbcTemplate.queryForObject("SELECT t.customer_nmb " +
                        " FROM dbo.t_creative t WITH ( NOLOCK ) " +
                        " JOIN dbo.t_creative_version cv WITH ( NOLOCK ) ON cv.creative_nmb = t.nmb " +
                        " WHERE cv.nmb = ?",
                Integer.class, creativeVersionId);

        LOGGER.info("retrieved customerNmb: {} for creativeVersionId: {}", customerNmb, creativeVersionId);

        // check if file exists already

        List<Integer> fileNmbs = jdbcTemplate.queryForList("SELECT nmb FROM dbo.t_file WHERE hash = ?",
                new Object[]{stillageAnswer.getMd5Hash()},
                new int[]{Types.BINARY},
                Integer.class);
        final Integer fileNmb;
        if (fileNmbs.isEmpty()) {
            LOGGER.info("existing file with given hash sum not found");

            final SimpleJdbcInsert jdbcInsertFile = new SimpleJdbcInsert(jdbcTemplate);
            jdbcInsertFile.withSchemaName("dbo")
                    .withTableName("t_file")
                    .usingColumns("hash", "stillage_file_id", "stillage_file_url", "stillage_file_metadata", "mime_type_nmb", "size")
                    .usingGeneratedKeyColumns("nmb");
            final Map<String, Object> parameters = new HashMap<>();
            parameters.put("hash", stillageAnswer.getMd5Hash());
            parameters.put("stillage_file_id", stillageAnswer.getId());
            parameters.put("stillage_file_url", stillageAnswer.getUrl());
            parameters.put("stillage_file_metadata", "{}");
            parameters.put("mime_type_nmb", 24); // TODO:  mime_type_nmb = 9 ?
            parameters.put("size", stillageAnswer.getFileSize());

            fileNmb = jdbcInsertFile.executeAndReturnKey(new MapSqlParameterSource(parameters)).intValue();
        } else {
            fileNmb = fileNmbs.iterator().next();
        }
        LOGGER.info("fileNmb: {}", fileNmb);

        final String fileName = "processed_code_file.html";

        Integer fileInstanceNmb = jdbcTemplate.queryForObject("SELECT MAX(nmb) " +
                        " FROM dbo.t_file_instance WHERE " +
                        " file_nmb = ? AND " +
                        " customer_nmb = ? AND " +
                        " [file_name] = ? AND " +
                        " [tag] = ?",
                Integer.class, fileNmb, customerNmb, fileName, "");
        if (fileInstanceNmb == null) {
            LOGGER.info("existing fileInstanceNmb not found");

            final SimpleJdbcInsert jdbcInsertFileInstance = new SimpleJdbcInsert(jdbcTemplate);
            jdbcInsertFileInstance.withSchemaName("dbo").withTableName("t_file_instance")
                    .usingColumns("file_nmb", "customer_nmb", "file_name", "tag", "file_added_date")
                    .usingGeneratedKeyColumns("nmb");

            fileInstanceNmb = jdbcInsertFileInstance.executeAndReturnKey(new MapSqlParameterSource(ImmutableMap.of(
                    "file_nmb", fileNmb,
                    "customer_nmb", customerNmb,
                    "file_name", fileName,
                    "tag", "",
                    "file_added_date", new Date()
            ))).intValue();
        }
        LOGGER.info("File content saved (CreativeVersionId: {}, FileInstanceId: {})", creativeVersionId, fileInstanceNmb);
        return fileInstanceNmb;
    }
}
