package ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.util.CollectionUtils;

import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.ChangedCreatives;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.ConstructorData;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.DbCreative;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.ParameterValue;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.PlaybackParameters;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.RtbHostMacroVideoConstructorData;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.VideoConstructorData;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.VideoFormat;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.services.CreativeService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbClientService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbIntegrationHealthService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.impl.BsApiClientException;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.impl.BsApiClientService;
import ru.yandex.bannerstorage.messaging.services.QueueMessageOnErrorStrategy;
import ru.yandex.bannerstorage.messaging.services.SimpleOneWayQueueObserver;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeExportEntry;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeParameterValue;
import ru.yandex.direct.bs.dspcreative.service.DspCreativeYtExporter;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

/**
 * @author santama
 */
public final class CreativeChangedQueueObserver extends SimpleOneWayQueueObserver<CreativeChangedRequest> {
    private static final String QUEUE_ID = "dbo.RtbHostLinkCreativeChangedServiceQueue";

    private static final Pattern EXTRACT_DOMAIN_REGEX = Pattern.compile("(:?.*://)?([^/]+)(:?/.*)?");

    /**
     * Размер 0х0 из таблицы t_smart_size
     */
    private static final int ADAPTIVE_SMART_SIZE_ID = 19;

    /**
     * Макет "Смарт-объявление" для смарт-ТГО.
     */
    private static final int SMART_TGO_LAYOUT_ID = 44;

    /**
     * Макеты для смарт-плитки
     */
    private static final Set<Integer> SMART_TILE_LAYOUTS = ImmutableSet.of(
            45, 46, 47, 48, 49, 50, 51, 52, 53
    );

    /**
     * Значения formatName для кодов, использованных в смартовых креативах,
     * созданных до появления параметра SMART_FORMAT_NAME
     */
    private static final Map<Integer, String> SMART_CODES_FORMATS = ImmutableMap.<Integer,String>builder()
            .put(807, "smart-banner_theme_big-image")
            .put(811, "smart-banner_theme_mosaic")
            .put(815, "smart-banner_theme_realty")
            .put(816, "smart-banner_theme_normal")
            .put(817, "smart-banner_theme_hover")
            .put(821, "smart-banner_theme_multiple")
            .put(831, "smart-banner_theme_travel")
            .put(843, "smart-banner_theme_auto")
            .put(844, "smart-banner_theme_avia")
            .put(845, "smart-banner_theme_auto-accordion")
            .put(846, "smart-banner_theme_auto-static-accordion")
            .put(867, "smart-banner_theme_clothes")
            // Для смарт-ТГО formatName не актуален: механизм показа смарт-ТГО принципиально отличается от
            // остальных смарт-баннеров, в них не используется JS-код, который обычно передаётся в Data
            .put(882, "")
            .put(883, "")
            .put(884, "")
            .put(885, "")
            .put(886, "")
            .put(964, "")
            .put(920, "smart-banner-adaptive_v1")
            .put(951, "smart-banner-adaptive_v1")
            .put(974, "smart-banner-mosaic_v1")
            .build();

    // Идентификаторы трекинговых событий
    // https://wiki.yandex-team.ru/bannerstorage/macro/
    public static final Map<String, String> AWAPS_EVENT_ID_TO_TRACKING_ACTION_ID = ImmutableMap.<String, String>builder()
            .putAll(ImmutableMap.of("52", "0", "54", "1", "53", "2", "55", "3", "56", "4"))
            .putAll(ImmutableMap.of("57", "5", "58", "6", "63", "9", "64", "13","89", "14")).build();
    public static final String AWAPS_EVENT_MACRO_PATTERN = "%AWAPS_AD_EVENT_URL\\((\\d+), \\d+\\)%";
    public static final String AWAPS_CLICK_URL_MACRO = "%AWAPS_AD_CLICK_URL%";
    public static final String RTB_CLICK_URL_MACRO = "\\${VAST_CLICK_URL}";
    public static final Pattern AWAPS_MACRO_PATTERN =
            Pattern.compile(AWAPS_EVENT_MACRO_PATTERN + "|" + AWAPS_CLICK_URL_MACRO);
    public static final Integer DIRECT_CUSTOMER_ID = 93;
    public static final String AWAPS_CLICK_URL_EVENT_ID = "1";

    public static final String CONSTRUCTOR_DATA_CODE = "CONSTRUCTOR_DATA_CODE";
    public static final String SMART_RENDER_PARAM_VALUES_CODE = "SMART_RENDER_PARAM_VALUES_CODE";
    public static final String STATIC_OBJ_CODE = "STATIC_OBJ_CODE";

    public static final String SMART_FORMAT_NAME = "SMART_FORMAT_NAME";

    private final CreativeService creativeService;
    private final RtbClientService rtbClientService;
    private final RtbIntegrationHealthService healthService;
    private final BsApiClientService bsApiClientService;
    private final DspCreativeYtExporter dspCreativeYtExporter;

    private final ObjectMapper objectMapper;

    public CreativeChangedQueueObserver(
            CreativeService creativeService,
            RtbClientService rtbClientService,
            RtbIntegrationHealthService healthService,
            BsApiClientService bsApiClientService,
            DspCreativeYtExporter dspCreativeYtExporter) {
        super(
                QUEUE_ID,
                DEFAULT_POLL_INTERVAL_IN_MS,
                5,
                5,
                QueueMessageOnErrorStrategy.INFINITE_POISON_USE_OLD_SESSION,
                CreativeChangedRequest.class);
        this.creativeService = Objects.requireNonNull(creativeService, "creativeService");
        this.rtbClientService = Objects.requireNonNull(rtbClientService, "rtbClientService");
        this.healthService = Objects.requireNonNull(healthService, "healthService");
        this.bsApiClientService = Objects.requireNonNull(bsApiClientService, "bsApiClientService");
        this.dspCreativeYtExporter = Objects.requireNonNull(dspCreativeYtExporter, "dspCreativeYtExporter");

        objectMapper = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    @Override
    public void doProcessMessage(@NotNull CreativeChangedRequest request) {
        try {
            doProcessMessageInternal(request);
        } catch (Throwable e) {
            getLogger().error("Exception when processing message", e);
            throw e;
        }
    }

    private void doProcessMessageInternal(@NotNull CreativeChangedRequest request) {
        if (null == request || CollectionUtils.isEmpty(request.getCreativeVersions())) {
            return;
        }

        List<Integer> creativeVersionIds = request.getCreativeVersions().stream()
                .map(CreativeChangedMessage::getCreativeVersionId)
                .collect(toList());

        ChangedCreatives changedCreatives = creativeService.getChangedCreatives(creativeVersionIds);

        List<DbCreative> dbCreatives = changedCreatives.getList();
        if (dbCreatives.isEmpty()) {
            return;
        }
        List<DspCreativeExportEntry> dspCreatives = new ArrayList<>();

        for (DbCreative dbCreative : dbCreatives) {
            DspCreativeExportEntry.Builder dspCreativeBuilder = createDspCreativeBuilder(dbCreative);

            setRequiredDefaults(dspCreativeBuilder);

            Integer id = dbCreative.getCreativeVersionId();
            dspCreativeBuilder.setTnsBrand(listIntToListLong(changedCreatives.getTnsBrands(id)));
            dspCreativeBuilder.setTnsArticle(listIntToListLong(changedCreatives.getTnsArticles(id)));
            dspCreativeBuilder.setGeo(listIntToListLong(changedCreatives.getGeo(id)));
            dspCreativeBuilder.setSite(listIntToListLong(changedCreatives.getSites(id)));
            Map<String, ParameterValue> parameterValues = changedCreatives.getParameters(id);
            if (parameterValues != null) {
                List<DspCreativeParameterValue> parameterValuesDsp = parameterValues.values()
                        .stream()
                        .map(CreativeChangedQueueObserver::mapParameterValueToDsp).collect(toList());
                dspCreativeBuilder.setParameterValues(parameterValuesDsp);
                final ParameterValue domainList = parameterValues.get("DOMAIN_LIST");
                if (domainList != null) {
                    dspCreativeBuilder.setDomainList(domainList.getParamValues().stream()
                            .map(d -> {
                                Matcher matcher = EXTRACT_DOMAIN_REGEX.matcher(d);
                                return matcher.find() ? matcher.group(2) : d;
                            }).collect(toList()));
                }
                Stream.of("IS_VIDEO", "IS_STATIC", "POSTMODERATED_OBJECTS")
                        .map(parameterValues::get)
                        .filter(Objects::nonNull)
                        .forEach(param -> Optional.ofNullable(param.getParamValues()).orElse(emptyList()).stream()
                            .filter(Objects::nonNull)
                            .filter(v -> v.matches("true|1"))
                            .forEach(b -> {
                                switch (param.getParamName()) {
                                    case "IS_VIDEO":
                                        dspCreativeBuilder.setVideo(true);
                                        break;
                                    case "IS_STATIC":
                                        dspCreativeBuilder.setStatic(true);
                                        break;
                                    case "POSTMODERATED_OBJECTS":
                                        dspCreativeBuilder.setPostmoderated(true);
                                        break;
                                }
                            }));
            }
            List<Pair<String, Integer>> showItems = changedCreatives.getShowItems(id);
            if (showItems != null) {
                for (Pair<String, Integer> param : showItems) {
                    switch (param.getKey()) {
                        case "FIRST_SHOW_ITEMS":
                            dspCreativeBuilder.setFirstShowItems(intToLong(param.getValue()));
                            break;
                        case "MIN_SHOW_ITEMS":
                            dspCreativeBuilder.setMinShowItems(intToLong(param.getValue()));
                            break;
                        case "MAX_SHOW_ITEMS":
                            dspCreativeBuilder.setMaxShowItems(intToLong(param.getValue()));
                            break;
                    }
                }
            }
            Pair<String, String> dspProp = changedCreatives.getDspProp(id);
            if (dspProp != null) {
                dspCreativeBuilder.setToken(dspProp.getRight());
            }

            if (isSmart(dbCreative)) {
                Integer minWidth = getIntParameterValueSafe(parameterValues, "MIN_WIDTH");
                Integer maxWidth = getIntParameterValueSafe(parameterValues, "MAX_WIDTH");
                Integer minHeight = getIntParameterValueSafe(parameterValues, "MIN_HEIGHT");
                Integer maxHeight = getIntParameterValueSafe(parameterValues, "MAX_HEIGHT");

                // Название формата креатива (для серверного рендеринга смартов)
                String formatName = getSmartFormatName(dbCreative, parameterValues);

                // Статическая часть сборки может шаблонизироваться прямо внутри хранимки
                // (если параметр задан на креативе) или его нужно шаблонизировать здесь
                // (если параметр фиксированный и привязан к коду нового движка)
                String staticData;
                if (dbCreative.isEnabled() && dbCreative.getStaticData().isEmpty()) {
                    try {
                        staticData = bsApiClientService.getCode(
                                dbCreative.getCreativeId(),
                                dbCreative.getCreativeVersionId(),
                                STATIC_OBJ_CODE
                        );
                        if (!isValidJson(staticData)) {
                            getLogger().error(String.format("Got invalid staticData for creative version %d",
                                    dbCreative.getCreativeVersionId()));
                            //
                            throw new RuntimeException("Got invalid staticData for performance creative");
                        }
                    } catch (BsApiClientException e) {
                        getLogger().error(String.format("Can't generate staticData for creative version %d",
                                dbCreative.getCreativeVersionId()), e);
                        throw e;
                    }
                } else {
                    staticData = dbCreative.getStaticData();
                }
                dspCreativeBuilder.setStaticData(staticData);

                // Шаблонизированные значения параметров (для серверного рендеринга)
                String renderedParamValues;
                if (dbCreative.isEnabled()) {
                    try {
                        renderedParamValues = bsApiClientService.getCode(
                                dbCreative.getCreativeId(),
                                dbCreative.getCreativeVersionId(),
                                SMART_RENDER_PARAM_VALUES_CODE
                        );
                        if (!isValidJson(renderedParamValues)) {
                            getLogger().error(String.format("Got invalid renderedParamValues JSON for creative version %d",
                                    dbCreative.getCreativeVersionId()));
                            //
                            renderedParamValues = null;
                        }
                    } catch (BsApiClientException e) {
                        getLogger().error(String.format("Can't get renderedParamValues for creative version %d",
                                dbCreative.getCreativeVersionId()), e);
                        //
                        renderedParamValues = null;
                    }
                } else {
                    // Если версия креатива отклонена, то мы не можем генерировать код, так как
                    // final-таргеты для файловых параметров не будут выполнены (они выполняются только при принятии креатива),
                    // и среди файлов на креативе может найтись такой, для которого ни разу не выполнялась
                    // final-конвертация (загрузка в MDS, к примеру)
                    // Из этих соображений в общем случае генерация production-кода для отклонённых креативов невозможна
                    renderedParamValues = null;
                }

                ConstructorData constructorData = new ConstructorData(
                        minWidth, maxWidth, minHeight, maxHeight,
                        SMART_TILE_LAYOUTS.contains(dbCreative.getSmartLayoutId()),
                        formatName,
                        renderedParamValues);
                dspCreativeBuilder.setConstructorData(toJson(constructorData));

                // Флаг адаптивности
                if (dbCreative.getSmartSizeId() == ADAPTIVE_SMART_SIZE_ID
                    && dbCreative.getSmartLayoutId() != SMART_TGO_LAYOUT_ID) {
                    dspCreativeBuilder.setIsAdaptive(true);
                }

                // Дополнительно устанавливаем themeId, layoutId, sizeId
                dspCreativeBuilder.setSmartThemeId(intToLong(dbCreative.getSmartThemeId()));
                dspCreativeBuilder.setSmartLayoutId(intToLong(dbCreative.getSmartLayoutId()));
                dspCreativeBuilder.setSmartSizeId(intToLong(dbCreative.getSmartSizeId()));
                //
            } else if (DIRECT_CUSTOMER_ID.equals(dbCreative.getCustomerId()) && dbCreative.isVideo()) {
                // Для видео креативов в Директе должно быть ConstructorData (см DIRECT-129123)
                // Без него креативы не будут работать на мобильном трафике
                try {
                    String constructorData = bsApiClientService.getCode(
                            dbCreative.getCreativeId(),
                            dbCreative.getCreativeVersionId(),
                            CONSTRUCTOR_DATA_CODE
                    );
                    //
                    getLogger().info("Got ConstructorData json for creative {}: {}", dbCreative.getCreativeId(), constructorData);
                    // Фронт складывает в параметр VPAID_CONSTRUCTOR_ELEMENTS макросы Авапса
                    // В основном коде мы их заменяем на аналоги в БК
                    // Код ConstructorData тоже может содержать этот параметр (и содержит для интерактивного шаблона)
                    // Поэтому замену нужно выполнить и здесь тоже
                    String rtbHostMacroConstructorData = replaceAwapsMacroWithRtbAnalog(constructorData);
                    getLogger().info("Prepared ConstructorData json for creative {}: {}",
                            dbCreative.getCreativeId(), rtbHostMacroConstructorData);
                    //
                    if (isValidJson(rtbHostMacroConstructorData)) {
                        dspCreativeBuilder.setConstructorData(rtbHostMacroConstructorData);
                        try {
                            RtbHostMacroVideoConstructorData data = objectMapper
                                    .readValue(rtbHostMacroConstructorData, RtbHostMacroVideoConstructorData.class);
                            if (data != null && data.getVideoFormats() != null) {
                                data.getVideoFormats().stream()
                                    .max(Comparator.comparingInt(VideoFormat::getWidth))
                                    .ifPresent(vf -> dspCreativeBuilder.setHeight(vf.getHeight()).setWidth(vf.getWidth()));
                            }
                        } catch (Exception e) {
                            getLogger().error("Failed to pars json from CONSTRUCTOR_DATA_CODE, creativeId = {}", dbCreative.getCreativeId(), e);
                        }
                    } else {
                        getLogger().error("Got invalid ConstructorData json for creative {}", dbCreative.getCreativeId());

                        // Пишем в RTBHost только duration (как раньше)
                        setDurationOnlyConstructorData(dspCreativeBuilder, parameterValues);
                    }
                } catch (BsApiClientException e) {
                    getLogger().error("Can't get ConstructorData for video creative in Direct", e);

                    // Пишем в RTBHost только duration (как раньше)
                    setDurationOnlyConstructorData(dspCreativeBuilder, parameterValues);
                }
            }

            dspCreatives.add(dspCreativeBuilder.build());
        }

        rtbClientService.notifyCreativeChanged(dspCreatives);

        dspCreativeYtExporter.export(dspCreatives);

        // Уведомляем что обработчик нашей очереди жив и нормально функционирует
        healthService.notifyAlive(RtbIntegrationHealthService.Queue.CREATIVE_CHANGED);
    }

    /**
     * Данная функция повторяет логику ручки bssoap, после ее выпиливания можно данную функцию убрать,
     * так как при отправке в YT данная логика уже реализована
     * https://a.yandex-team.ru/arc/trunk/arcadia/yabs/interface/yabs-bssoap-export/cgi/import-dsp-creative.cgi?blame=true&rev=r7527674#L143
     */
    private static void setRequiredDefaults(DspCreativeExportEntry.Builder dspCreativeEntry) {
        dspCreativeEntry
                .setStaticData("")
                .setMaxShowItems(0L)
                .setMinShowItems(0L)
                .setFirstShowItems(0L)
                .setDomainList(emptyList())
                .setSmartThemeId(0L)
                .setSmartLayoutId(0L)
                .setSmartSizeId(0L);
    }

    /**
     * Получение formatName для смартового креатива
     * Если на креативе есть соответствующий параметр, то formatName берётся из него
     * Если параметра нет, то значение определяется из номера кода
     * (это нужно для креативов, созданных до того, как новый параметр был добавлен в шаблоны)
     */
    String getSmartFormatName(DbCreative dbCreative, Map<String, ParameterValue> parameterValues) {
        String result;
        String smartFormatName = getStringParameterValueSafe(parameterValues, SMART_FORMAT_NAME);
        if (smartFormatName != null) {
            result = smartFormatName;
        } else {
            if (SMART_CODES_FORMATS.containsKey(dbCreative.getCodeId())) {
                result = SMART_CODES_FORMATS.get(dbCreative.getCodeId());
            } else {
                // Для смартов нужно обязательно уметь определять formatName
                // Если мы этого сделать не можем, значит, шаблон был настроен неправильно
                getLogger().error("Can't get formatName for creative version {}", dbCreative.getCreativeVersionId());
                throw new RuntimeException("Can't get formatName for creative");
            }
        }
        // Пустая строка означает, что формата фактически нет
        return result.isEmpty() ? null : result;
    }

    private void setDurationOnlyConstructorData(DspCreativeExportEntry.Builder rtbCreative,
                                                Map<String, ParameterValue> parameterValues) {
        // Лечим https://st.yandex-team.ru/DIRECT-125955
        // Интерактивный дисплейный шаблон в Директе умеет передавать длительность рекламного ролика?
        // У нас крутятся 30-секундные ролики на блоке с ограничением 15 сек
        Integer duration = getIntParameterValueSafe(parameterValues, "DURATION");
        // У старых креативов нет ConstructorData, но поле playbackParameters обязательно должно быть
        // У всех шаблонов есть кнопка, поэтому смело выставляем showSkipButton=true
        PlaybackParameters playbackParameters = new PlaybackParameters().withShowSkipButton(true);
        VideoConstructorData videoConstructorData = new VideoConstructorData(duration, playbackParameters);
        // а откуда длительность креатива берется?
        // Но вообще берётся из https://yt.yandex-team.ru/hahn/navigation?offsetMode=key&path=//home/yabs/dict/DSPCreative конструктор даты
        // в корне ConstructorData duration
        // https://a.yandex-team.ru/arc/trunk/arcadia/yabs/server/cs/libs/mkdb/dsp_creative.cpp?blame=true&rev=7185957#L410
        rtbCreative.setConstructorData(toJson(videoConstructorData));
    }

    private boolean isValidJson(String json) {
        try {
            JsonParser parser = new ObjectMapper().getFactory().createParser(json);
            while (parser.nextToken() != null) {
                // do nothing
            }
            return true;
        } catch (IOException e) {
            return false;
        }
    }

    private boolean isSmart(DbCreative dbCreative) {
        return dbCreative.getSmartSizeId() != null;
    }

    private Integer getIntParameterValueSafe(@Nullable Map<String, ParameterValue> parameterValues,
                                             String paramName) {
        if (parameterValues == null) {
            return null;
        }
        ParameterValue parameterValue = parameterValues.get(paramName);
        if (parameterValue == null) {
            return null;
        }
        if (!"int".equals(parameterValue.getParamType())) {
            return null;
        }
        if (parameterValue.getParamValues().size() != 1) {
            return null;
        }
        if (parameterValue.getParamValues().get(0) == null) {
            return null;
        }
        try {
            return Integer.parseInt(parameterValue.getParamValues().get(0));
        } catch (NumberFormatException e) {
            return null;
        }
    }

    private String getStringParameterValueSafe(@Nullable Map<String, ParameterValue> parameterValues,
                                               String paramName) {
        if (parameterValues == null) {
            return null;
        }
        ParameterValue parameterValue = parameterValues.get(paramName);
        if (parameterValue == null) {
            return null;
        }
        if (!"string".equals(parameterValue.getParamType())) {
            return null;
        }
        if (parameterValue.getParamValues().size() != 1) {
            return null;
        }
        if (parameterValue.getParamValues().get(0) == null) {
            return null;
        }
        return parameterValue.getParamValues().get(0);
    }

    /**
     * Создаёт полузаполненную модель креатива для отправки в RTBHost на основе данных из базы
     * Заполняются только те поля, которые мапятся как есть. Остальное дозаполняется отдельно.
     */
    private DspCreativeExportEntry.Builder createDspCreativeBuilder(DbCreative dbCreative) {
        DspCreativeExportEntry.Builder dspCreativeBuilder = DspCreativeExportEntry.builder();
        dspCreativeBuilder.setCreativeId(intToLong(dbCreative.getCreativeId()));
        dspCreativeBuilder.setCreativeVersionId(intToLong(dbCreative.getCreativeVersionId()));
        dspCreativeBuilder.setCreativeTemplateId(intToLong(dbCreative.getTemplateId()));
        dspCreativeBuilder.setCreativeCodeId(intToLong(dbCreative.getCodeId()));
        dspCreativeBuilder.setDspId(intToLong(dbCreative.getDspId()));
        dspCreativeBuilder.setWidth(dbCreative.getWidth());
        dspCreativeBuilder.setHeight(dbCreative.getHeight());
        dspCreativeBuilder.setExpireTime(dbCreative.getExpireTime());
        dspCreativeBuilder.setCreateTime(dbCreative.getCreateTime());
        dspCreativeBuilder.setEnabled(dbCreative.isEnabled());
        // временное решение до https://st.yandex-team.ru/DIRECT-115417
        // [bs-direct][2] AdDesigner убрать хардкод авапсных макросов из фронта
        if (DIRECT_CUSTOMER_ID.equals(dbCreative.getCustomerId()) && dbCreative.isVideo()) {
            String rtbReadyData = replaceAwapsMacroWithRtbAnalog(dbCreative.getData());
            dspCreativeBuilder.setData(rtbReadyData);
        } else {
            dspCreativeBuilder.setData(dbCreative.getData());
        }
        dspCreativeBuilder.setStaticData(dbCreative.getStaticData());
        dspCreativeBuilder.setVideo(dbCreative.isVideo());
        dspCreativeBuilder.setTag(dbCreative.getRtbTag());
        return dspCreativeBuilder;
    }

    private static Long intToLong(@Nullable Integer value) {
        return value == null ? null : Long.valueOf(value);
    }

    private static List<Long> listIntToListLong(@Nullable List<Integer> listInt) {
        if (listInt == null) {
            return null;
        }
        return listInt.stream().map(CreativeChangedQueueObserver::intToLong).collect(Collectors.toList());
    }

    private static DspCreativeParameterValue mapParameterValueToDsp(@Nullable ParameterValue parameterValue) {
        if (parameterValue == null) {
            return null;
        }
        return new DspCreativeParameterValue(
                parameterValue.getParamName(), parameterValue.getParamValues(),
                parameterValue.getOrderInList(), parameterValue.getParamType());
    }

    @NotNull
    public static String replaceAwapsMacroWithRtbAnalog(String data) {
        Matcher matcher = AWAPS_MACRO_PATTERN.matcher(data);
        StringBuffer sb = new StringBuffer();
        while (matcher.find()) {
            String match = matcher.group();
            if (AWAPS_CLICK_URL_MACRO.equals(match)) {
                matcher.appendReplacement(sb, RTB_CLICK_URL_MACRO);
            } else {
                String eventId = matcher.group(1);
                String actionId = AWAPS_EVENT_ID_TO_TRACKING_ACTION_ID.get(eventId);
                if (actionId != null) {
                    matcher.appendReplacement(sb, "\\${TRACKING_URL_PREFIX}?action-id=" + actionId);
                } else if (AWAPS_CLICK_URL_EVENT_ID.equals(eventId)) {
                    // макрос AWAPS '%AWAPS_AD_EVENT_URL(1, x)%' соответствует '${VAST_CLICK_URL}' в БК
                    matcher.appendReplacement(sb, RTB_CLICK_URL_MACRO);
                } else {
                    // если отображение события AWAPS на событие БК не возможно
                    // заменяем макрос на пустую строку
                    matcher.appendReplacement(sb, "");
                }
            }
        }
        matcher.appendTail(sb);

        return sb.toString();
    }

    private String toJson(Object o) {
        try {
            return objectMapper.writeValueAsString(o);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }
}
