package ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.services.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.validation.constraints.NotNull;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.jdbc.core.CallableStatementCreator;
import org.springframework.jdbc.core.CallableStatementCreatorFactory;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.SqlParameter;
import org.springframework.jdbc.core.SqlReturnResultSet;

import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.ChangedCreatives;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.DbCreative;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.models.ParameterValue;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.creativechanged.services.CreativeService;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

/**
 * @author egorovmv
 */
public final class JdbcCreativeService implements CreativeService {
    private final JdbcTemplate jdbcTemplate;

    // The input parameters of the stored procedure
    private final List<SqlParameter> declaredParams = singletonList(
            new SqlParameter("creative_version_nmbs_list", Types.VARCHAR));

    private final CallableStatementCreatorFactory cscFactory = new CallableStatementCreatorFactory(
            "{call dbo.sp_rtbhost_creative_changed_getList(@creative_version_nmbs_list=?)}", declaredParams);

    private static final RowMapper<Pair<Integer, Integer>> PAIR_ROW_MAPPER =
            (RowMapper<Pair<Integer, Integer>>) (rs, rowNum) -> Pair.of(rs.getInt(1), rs.getInt(2));

    private static Integer readInteger(ResultSet rs, String column) throws SQLException {
        int value = rs.getInt(column);
        return rs.wasNull() ? null : value;
    }

    // The result sets of the stored procedure
    private static final List<SqlParameter> returnedParams = Arrays.<SqlParameter>asList(
            new SqlReturnResultSet("creative_versions", (rs, rowNum) -> {
                final DbCreative creative = new DbCreative();
                creative.setCreativeId(rs.getInt("creativeId"));
                creative.setCreativeVersionId(rs.getInt("creativeVersionId"));
                creative.setTemplateId(rs.getInt("templateId"));
                creative.setCodeId(rs.getInt("codeId"));
                creative.setCustomerId(rs.getInt("customerId"));
                creative.setCustomerName(rs.getString("customerName"));
                creative.setDspId(rs.getInt("dspId"));
                creative.setWidth(rs.getInt("width"));
                creative.setHeight(rs.getInt("height"));
                creative.setUpdateTime(rs.getDate("updateTime"));
                creative.setExpireTime(rs.getDate("expireTime"));
                creative.setCreateTime(rs.getDate("createTime"));
                creative.setEnabled(rs.getBoolean("isEnabled"));
                creative.setData(rs.getString("data"));
                creative.setStaticData(rs.getString("staticData"));
                creative.setVideo(rs.getBoolean("isVast"));
                creative.setOrderNo(rs.getInt("orderNo"));
                creative.setRtbTag(rs.getString("rtbTag"));
                creative.setSmartThemeId(readInteger(rs, "smartThemeId"));
                creative.setSmartSizeId(readInteger(rs, "smartSizeId"));
                creative.setSmartLayoutId(readInteger(rs, "smartLayoutId"));
                return creative;
            }),
            new SqlReturnResultSet("tns_brands", PAIR_ROW_MAPPER),
            new SqlReturnResultSet("tns_articles", PAIR_ROW_MAPPER),
            new SqlReturnResultSet("geo", PAIR_ROW_MAPPER),
            new SqlReturnResultSet("sites", PAIR_ROW_MAPPER),
            new SqlReturnResultSet("show_items", (rs, rowNum) ->
                    Pair.of(rs.getInt(1), Pair.of(rs.getString(2), rs.getInt(3)))),
            new SqlReturnResultSet("parameters", (rs, rowNum) ->
                    Pair.of(rs.getInt(1),
                            new ParameterValue(
                                    rs.getString(2),
                                    new ArrayList<>(singletonList(rs.getString(3))),
                                    rs.getInt(4),
                                    rs.getString(5)))),
            new SqlReturnResultSet("dsp_props", (rs, rowNum) ->
                    Pair.of(rs.getInt(1), Pair.of(rs.getString(2), rs.getString(3)))));

    public JdbcCreativeService(@NotNull JdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = Objects.requireNonNull(jdbcTemplate, "jdbcTemplate");
    }

    @Override
    public ChangedCreatives getChangedCreatives(@NotNull List<Integer> creativeVersionIds) {
        // query database
        final Map<String, Object> actualParams = Collections.singletonMap("creative_version_nmbs_list",
                creativeVersionIds.stream().map(Object::toString).collect(joining(",")));
        final CallableStatementCreator csc = cscFactory.newCallableStatementCreator(actualParams);
        final Map<String, Object> result = jdbcTemplate.call(csc, returnedParams);

        // map result sets to creative objects
        @SuppressWarnings("unchecked")
        final List<DbCreative> creatives = (List<DbCreative>) result.get("creative_versions");
        @SuppressWarnings("unchecked")
        final Map<Integer, List<Integer>> tnsBrands = ((List<Pair<Integer, Integer>>) result.get("tns_brands")).stream().collect(groupingBy(Pair::getKey, mapping(Pair::getValue, toList())));
        @SuppressWarnings("unchecked")
        final Map<Integer, List<Integer>> tnsArticles = ((List<Pair<Integer, Integer>>) result.get("tns_articles")).stream().collect(groupingBy(Pair::getKey, mapping(Pair::getValue, toList())));
        @SuppressWarnings("unchecked")
        final Map<Integer, List<Integer>> geo = ((List<Pair<Integer, Integer>>) result.get("geo")).stream().collect(groupingBy(Pair::getKey, mapping(Pair::getValue, toList())));
        @SuppressWarnings("unchecked")
        final Map<Integer, List<Integer>> sites = ((List<Pair<Integer, Integer>>) result.get("sites")).stream().collect(groupingBy(Pair::getKey, mapping(Pair::getValue, toList())));
        @SuppressWarnings("unchecked")
        final Map<Integer, Map<String, ParameterValue>> parameters =
                ((List<Pair<Integer, ParameterValue>>) result.get("parameters")).stream()
                        .sorted(Comparator.comparing(pair -> ((Integer) ((Pair) pair).getKey()))
                                .thenComparing(pair -> ((ParameterValue) ((Pair) pair).getValue()).getParamName())
                                .thenComparing(pair -> ((ParameterValue) ((Pair) pair).getValue()).getOrderInList()))
                        .collect(groupingBy(Pair::getKey,
                                toMap(p -> p.getValue().getParamName(), Pair::getValue,
                                        (p1, p2) -> {
                                            p1.getParamValues().addAll(p2.getParamValues());
                                            return p1;
                                        })));
        @SuppressWarnings("unchecked")
        final Map<Integer, List<Pair<String, Integer>>> showItems =
                ((List<Pair<Integer, Pair<String, Integer>>>) result.get("show_items")).stream()
                        .collect(groupingBy(Pair::getKey, mapping(Pair::getValue, toList())));
        @SuppressWarnings("unchecked")
        final Map<Integer, Pair<String, String>> dspProps =
                ((List<Pair<Integer, Pair<String, String>>>) result.get("dsp_props")).stream()
                        .collect(toMap(Pair::getKey, Pair::getValue, (p1, p2) -> p1));

        return new ChangedCreatives(
                creatives, tnsBrands, tnsArticles, geo, sites, parameters, showItems, dspProps);
    }
}
