package ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.impl;

import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbClientErrorException;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbClientService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.ModeratedOffer;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.UnmoderatedAssembly;
import ru.yandex.bannerstorage.harvester.utils.JsonUtils;
import ru.yandex.bannerstorage.harvester.utils.RestTemplateFactory;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeExportEntry;

/**
 * @author egorovmv
 */
public final class HttpRtbClientService implements RtbClientService {
    private static final Logger logger = LoggerFactory.getLogger(HttpRtbClientService.class);

    private final RestTemplate restTemplate;
    private volatile URI serviceUrl;
    private final ObjectWriter changedCreativesWriter;
    private final ObjectReader unmoderatedAssembliesReader;
    private final ObjectWriter moderatedOffersWriter;

    public HttpRtbClientService(
            @NotNull RestTemplateFactory restTemplateFactory,
            @NotNull String serviceUrl,
            int connectTimeoutInMS,
            int readTimeoutInMS) {
        Objects.requireNonNull(restTemplateFactory);
        Objects.requireNonNull(serviceUrl);
        if (connectTimeoutInMS <= 0)
            throw new IllegalArgumentException("connectTimeoutInMS: " + connectTimeoutInMS);
        if (readTimeoutInMS <= 0)
            throw new IllegalArgumentException("readTimeoutInMS: " + readTimeoutInMS);

        logger.info(
                "Constructing (serviceUrl: {}, connectTimeoutInMS: {}, readTimeoutInfMS: {})...",
                serviceUrl, connectTimeoutInMS, readTimeoutInMS);

        this.restTemplate = restTemplateFactory.create(connectTimeoutInMS, readTimeoutInMS);

        this.serviceUrl = URI.create(serviceUrl);

        ObjectMapper jsonMapper = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

        this.changedCreativesWriter = jsonMapper.writerFor(
                new TypeReference<List<DspCreativeExportEntry>>() {
                });
        this.unmoderatedAssembliesReader = jsonMapper.readerFor(
                new TypeReference<List<UnmoderatedAssembly>>() {
                });
        this.moderatedOffersWriter = new ObjectMapper().writerFor(
                new TypeReference<List<ModeratedOffer>>() {
                });

        logger.info("Constructed");
    }

    @Override
    public void notifyCreativeChanged(@NotNull List<DspCreativeExportEntry> creatives) {
        URI targetUrl = UriComponentsBuilder.fromUri(serviceUrl)
                .pathSegment("import-dsp-creative.cgi")
                .build()
                .toUri();

        UUID requestId = UUID.randomUUID();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
        headers.setAccept(
                Collections.singletonList(
                        MediaType.valueOf(MediaType.ALL_VALUE)));

        try {
            MultiValueMap<String, Object> requestBody = new LinkedMultiValueMap<>();
            String creativesJson = JsonUtils.serialize(changedCreativesWriter, creatives);
            requestBody.add("request", creativesJson);

            logger.info(
                    "Sending request (RequestId: {}, TargetUrl: {}, Creatives: {})",
                    requestId, targetUrl, creativesJson);

            ResponseEntity<String> responseEntity = restTemplate.exchange(
                    targetUrl,
                    HttpMethod.POST,
                    new HttpEntity<>(requestBody, headers),
                    String.class);

            String response = responseEntity.getBody();

            logger.info(
                    "Got response (RequestId: {}, Response: {}, Status: {})",
                    requestId, response, responseEntity.getStatusCode());
        } catch (ResourceAccessException | HttpStatusCodeException e) {
            if (e instanceof HttpClientErrorException) {
                logger.error("Got response on failed request: {}",
                        ((HttpClientErrorException) e).getResponseBodyAsString());
            }
            throw new RtbClientErrorException(requestId, !(e instanceof HttpClientErrorException), e);
        }
    }

    @Override
    public List<UnmoderatedAssembly> getUnmoderatedAssemblies(@NotNull Integer fromQueueId, int fetchCount) {
        URI targetUrl = UriComponentsBuilder.fromUri(serviceUrl)
                .pathSegment("export_dsp-moderation-queue.cgi")
                .queryParam("lastid", fromQueueId)
                .queryParam("count", Integer.toString(fetchCount))
                .build()
                .toUri();

        UUID requestId = UUID.randomUUID();

        try {
            logger.info(
                    "Sending request (RequestId: {}, TargetUrl: {})",
                    requestId, targetUrl);

            ResponseEntity<String> responseEntity = restTemplate.exchange(
                    targetUrl,
                    HttpMethod.GET,
                    null,
                    String.class);

            String response = responseEntity.getBody();

            logger.info("Got response (RequestId: {}, Response: {})", requestId, response);

            return JsonUtils.deserialize(unmoderatedAssembliesReader, response);
        } catch (ResourceAccessException | HttpStatusCodeException e) {
            throw new RtbClientErrorException(requestId, !(e instanceof HttpClientErrorException), e);
        }
    }

    @Override
    public void sendModeratedOffers(@NotNull List<ModeratedOffer> offers) {
        URI targetUrl = UriComponentsBuilder.fromUri(serviceUrl)
                .pathSegment("import_dsp-moderation-result.cgi")
                .build()
                .toUri();

        UUID requestId = UUID.randomUUID();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
        headers.setAccept(
                Collections.singletonList(
                        MediaType.valueOf(MediaType.ALL_VALUE)));

        try {
            logger.info(
                    "Sending request (RequestId: {}, TargetUrl: {}, Request: {})",
                    requestId, targetUrl, offers);

            MultiValueMap<String, Object> requestBody = new LinkedMultiValueMap<>();
            requestBody.add("request", JsonUtils.serialize(moderatedOffersWriter, offers));

            ResponseEntity<String> responseEntity = restTemplate.exchange(
                    targetUrl,
                    HttpMethod.POST,
                    new HttpEntity<>(requestBody, headers),
                    String.class);

            String response = responseEntity.getBody();

            logger.info("Got response (RequestId: {}, Response: {})", requestId, response);
        } catch (ResourceAccessException | HttpStatusCodeException e) {
            throw new RtbClientErrorException(requestId, !(e instanceof HttpClientErrorException), e);
        }
    }

    public void setServiceUrl(String serviceUrl) {
        this.serviceUrl = URI.create(serviceUrl);
    }
}
