package ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;

import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbClientErrorException;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbClientService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbIntegrationHealthService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.exceptions.InvalidCreativeVersionException;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.exceptions.OfferDcParamsNotFoundException;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.AssemblyCreativeMaps;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.DcParams;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.UnmoderatedAssembly;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.UnmoderatedOffer;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.services.AssemblyService;
import ru.yandex.bannerstorage.harvester.utils.JsonUtils;
import ru.yandex.bannerstorage.messaging.services.TimerQueueObserver;
import ru.yandex.bannerstorage.messaging.services.exceptions.AbortMessageProcessingException;

import static java.util.stream.Collectors.toList;

/**
 * Очередь для получения неотмодерированных offer-ов из БК для постмодерации
 *
 * @author egorovmv
 */
public class RecvUnmoderatedAssembliesQueueObserver extends TimerQueueObserver {
    private static final String QUEUE_ID = "PostModeration.RecvUnmoderatedAssembliesQueue";

    private final AssemblyService assemblyService;
    private final RtbClientService rtbClient;
    private final RtbIntegrationHealthService healthService;
    private final int fetchCount;

    private final ObjectReader dcParamsJsonReader;

    public RecvUnmoderatedAssembliesQueueObserver(
            @NotNull AssemblyService assemblyService,
            @NotNull RtbClientService rtbClient,
            @NotNull RtbIntegrationHealthService healthService,
            int fetchCount,
            int pollIntervalInMS) {
        super(QUEUE_ID, pollIntervalInMS);

        Objects.requireNonNull(assemblyService, "assemblyService");
        Objects.requireNonNull(rtbClient, "rtbClient");
        Objects.requireNonNull(healthService, "healthService");
        if (fetchCount <= 0)
            throw new IllegalArgumentException("fetchCount: " + fetchCount);

        this.assemblyService = assemblyService;
        this.rtbClient = rtbClient;
        this.healthService = healthService;
        this.fetchCount = fetchCount;

        this.dcParamsJsonReader = new ObjectMapper().readerFor(new TypeReference<DcParams>() {
        });
    }

    private List<UnmoderatedAssembly> getUnmoderatedAssemblies(
            @NotNull Integer fromQueueId, int fetchCount) {
        try {
            return rtbClient.getUnmoderatedAssemblies(fromQueueId, fetchCount);
        } catch (RtbClientErrorException e) {
            // Какие-то проблемы с БК, надо прервать обработку сообщений
            // и возможно как-то ее перепланировать
            if (e.isRecoverable())
                throw new AbortMessageProcessingException(e);
            else
                throw e;
        }
    }

    private void saveAssembly(@NotNull AssemblyCreativeMaps creativeMaps, @NotNull UnmoderatedAssembly assembly) {
        Integer creativeVersionId = creativeMaps.getVersionId(
                assembly.getAssembly().getCreativeId());
        // Если мы не нашли соотвествующую креативу версию, то это не наш креатив,
        // поэтому мы его пропускаем
        if (creativeVersionId == null)
            return;

        // Если в сборке задана версия, то она должна соответствовать креативу
        if (assembly.getAssembly().getCreativeVersionId() != null
                && creativeMaps.getCreativeId(assembly.getAssembly().getCreativeVersionId()) == null) {
            throw new InvalidCreativeVersionException(
                    assembly.getAssembly().getCreativeVersionId());
        }

        Integer assemblyId = assemblyService.persistAssembly(assembly.getAssembly(), creativeVersionId);

        // Если идентификатор сборки равен null, то такая сборка уже была записана и должна быть пропущена
        if (assemblyId == null)
            return;

        // Приводим параметры отображания всех offer-ов к единому виду
        Map<String, Map<String, Object>> dcParams = JsonUtils.<DcParams>deserialize(
                dcParamsJsonReader, assembly.getDcParams()).toMap();

        for (int i = 0; i < assembly.getOffers().size(); i++) {
            UnmoderatedOffer offer = assembly.getOffers().get(i);

            Map<String, Object> offerDcParams = dcParams.get(offer.getKeyName());
            if (offerDcParams == null) {
                throw new OfferDcParamsNotFoundException(
                        assembly.getAssembly().getQueueId(), offer.getKeyName());
            }

            assemblyService.persistOffer(
                    assemblyId, offer, dcParams.get(offer.getKeyName()), i + 1);
        }
    }

    @Override
    public Map<String, Object> newLocalState() {
        return new HashMap<>();
    }

    private Integer getLatestQueueId(@NotNull Map<String, Object> localState) {
        Objects.requireNonNull(localState, "localState");
        Integer latestQueueId = assemblyService.getLatestQueueId();
        if (latestQueueId == null)
            latestQueueId = 0;
        Integer savedLatestQueueId = (Integer)localState.getOrDefault("latestQueueId", 0);
        return savedLatestQueueId > latestQueueId ? savedLatestQueueId : latestQueueId;
    }

    private void moveLatestQueueIdTo(@NotNull Map<String, Object> localState, @NotNull Integer newQueueId) {
        Objects.requireNonNull(localState, "localState");
        Objects.requireNonNull(newQueueId, "newQueueId");
        localState.put("latestQueueId", newQueueId);
    }

    @Override
    protected boolean OnTimer(Map<String, Object> localState) {
        // Прочитать указатель на последний считанный queueId сборки
        Integer latestQueueId = getLatestQueueId(localState);

        // Прочитать сборки из БК начиная с заданного queueId (исключая его)
        List<UnmoderatedAssembly> assemblies = getUnmoderatedAssemblies(latestQueueId, fetchCount);

        if (!assemblies.isEmpty()) {
            AssemblyCreativeMaps creativeMaps = assemblyService.getAssemblyCreativeMaps(
                    assemblies.stream()
                            .map(a -> a.getAssembly().getCreativeId())
                            .collect(toList()),
                    assemblies.stream()
                            .map(a -> a.getAssembly().getCreativeVersionId())
                            .filter(Objects::nonNull)
                            .collect(toList()));

            for (UnmoderatedAssembly assembly : assemblies){
                saveAssembly(creativeMaps, assembly);
            }

            // Двигаем указатель дальше для того, чтобы не крутится в бесконечном цикле
            moveLatestQueueIdTo(
                    localState,
                    assemblies.stream()
                            .mapToInt(a -> a.getAssembly().getQueueId())
                            .max()
                            .getAsInt());
        }

        // Уведомляем что обработчик нашей очереди жив и нормально функционирует
        healthService.notifyAlive(RtbIntegrationHealthService.Queue.RECV_UNMODERATED_ASSEMBLIES);

        // Пока мы что-то извлекли, мы должны повторять обработку,
        // но уже в другом цикле обработки сообщений (будет другая транзакция),
        // иначе слишком долго будут держаться блокировки
        return !assemblies.isEmpty();
    }
}
