package ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.validation.constraints.NotNull;

import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbClientErrorException;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbClientService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.infrastructure.RtbIntegrationHealthService;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.ModeratedOffer;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.services.AssemblyService;
import ru.yandex.bannerstorage.messaging.services.TimerQueueObserver;
import ru.yandex.bannerstorage.messaging.services.exceptions.AbortMessageProcessingException;

/**
 * @author egorovmv
 */
public final class SendModeratedOffersQueryObserver extends TimerQueueObserver {
    private static final String QUEUE_ID = "PostModeration.SendModeratedOffersQueue";

    private final AssemblyService assemblyService;
    private final RtbClientService rtbClient;
    private final RtbIntegrationHealthService healthService;
    private final int fetchCount;

    public SendModeratedOffersQueryObserver(
            @NotNull AssemblyService assemblyService,
            @NotNull RtbClientService rtbClient,
            @NotNull RtbIntegrationHealthService healthService,
            int fetchCount,
            int pollIntervalInMS) {
        super(QUEUE_ID, pollIntervalInMS);
        this.assemblyService = Objects.requireNonNull(assemblyService, "assemblyService");
        this.rtbClient = Objects.requireNonNull(rtbClient, "rtbClient");
        this.healthService = Objects.requireNonNull(healthService, "healthService");
        this.fetchCount = fetchCount;
    }

    @Override
    protected boolean OnTimer(Map<String, Object> localState) {
        // Извлечь отмодерированные offer-ы, которые еще не отправлены в БК,
        // из очереди (в данном случае фиктивной)
        List<ModeratedOffer> offers = assemblyService.getUnsyncModeratedOffers(fetchCount);

        if (!offers.isEmpty()) {
            // Отправить отмодерированные offer-ы в БК
            try {
                rtbClient.sendModeratedOffers(offers);
            } catch (RtbClientErrorException e) {
                // Какие-то проблемы с БК, надо прервать обработку сообщений
                // и возможно как-то ее перепланировать
                if (e.isRecoverable())
                    throw new AbortMessageProcessingException(e);
                else
                    throw e;
            }

            // Помечаем offer-ы как отправленные в БК
            assemblyService.markOffersAsSync(
                    offers.stream()
                            .map(ModeratedOffer::getObjectId)
                            .collect(Collectors.toList()));
        }

        // Уведомляем что обработчик нашей очереди жив и нормально функционирует
        healthService.notifyAlive(RtbIntegrationHealthService.Queue.SEND_MODERATED_OFFERS);

        // Пока мы что-то извлекли, мы должны повторять обработку,
        // но уже в другом цикле обработки сообщений (будет другая транзакция),
        // иначе слишком долго будут держаться блокировки
        return !offers.isEmpty();
    }
}
