package ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models;

import java.math.BigInteger;
import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.bannerstorage.harvester.utils.ZoneOffsets;

/**
 * Десериализованная сборка в том виде в котором она непосредственно приходит из БК
 *
 * @author egorovmv
 */
public final class UnmoderatedAssembly {
    private static final DateTimeFormatter YYYY_MM_DD_HH_MM_SS = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private static final DateTimeFormatter YYYYMMDDHHMMSS = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");

    private final AssemblyInfo assembly;
    private final List<UnmoderatedOffer> offers;
    private final String dcParams;

    @JsonCreator
    public UnmoderatedAssembly(
            @JsonProperty(value = "QueueID", required = true) @NotNull Integer queueId,
            @JsonProperty(value = "CreativeID", required = true) @NotNull Integer creativeId,
            @JsonProperty("CreativeVersionID") Integer creativeVersionId,
            @JsonProperty("Creative") CreativeInfo creative,
            @JsonProperty("BidReqID") BigInteger bidReqId,
            @JsonProperty("BidReqTime") String bidReqTime, // "yyyy-MM-dd HH:mm:ss"
            @JsonProperty("RegionID") Integer regionId,
            @JsonProperty("LastKnownExpression") String lastKnownImpression, // "yyyy-MM-dd HH:mm:ss"
            @JsonProperty("NumberOfExpressions") Integer numberOfImpressions,
            @JsonProperty(value = "Enqueued", required = true) String enqueued, // "yyyyMMddHHmmss"
            @JsonProperty("Width") Integer width,
            @JsonProperty("Height") Integer height,
            @JsonProperty(value = "objects", required = true) List<UnmoderatedOffer> offers,
            @JsonProperty(value = "dc_params", required = true) @NotNull String dcParams) {
        this.assembly = new AssemblyInfo(
                queueId,
                creativeId,
                creativeVersionId,
                // bidReqId приходит как BigInteger, но в БД мы его сохраняем как строку
                bidReqId != null ? bidReqId.toString() : null,
                bidReqTime != null ? Timestamp.from(LocalDateTime.parse(bidReqTime, YYYY_MM_DD_HH_MM_SS).toInstant(ZoneOffsets.MOSCOW)) : null,
                regionId,
                creative != null ? (creative.getHits() != null ? creative.getHits() : 1) : 1,
                lastKnownImpression != null
                        ? Timestamp.from(LocalDateTime.parse(lastKnownImpression, YYYY_MM_DD_HH_MM_SS).toInstant(ZoneOffsets.MOSCOW))
                        : null,
                numberOfImpressions,
                Timestamp.from(
                        LocalDateTime.parse(
                                Objects.requireNonNull(enqueued, "enqueued"), YYYYMMDDHHMMSS).toInstant(ZoneOffsets.MOSCOW)),
                width,
                height);
        this.offers = offers != null ? offers : Collections.emptyList();
        this.dcParams = Objects.requireNonNull(dcParams, "dcParams");
    }

    @Override
    public String toString() {
        return "UnmoderatedAssembly{" +
                "assembly=" + assembly +
                // ", offers=" + offers + Специально закомментированно, чтобы в логи не выводились
                ", dcParams='" + dcParams + '\'' +
                '}';
    }

    public AssemblyInfo getAssembly() {
        return assembly;
    }

    public List<UnmoderatedOffer> getOffers() {
        return offers;
    }

    public String getDcParams() {
        return dcParams;
    }
}
