package ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.services.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.CallableStatementCreatorFactory;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.SqlParameter;
import org.springframework.jdbc.core.SqlReturnResultSet;
import org.springframework.jdbc.core.namedparam.BeanPropertySqlParameterSource;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;

import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.AssemblyCreativeMaps;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.AssemblyInfo;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.ModeratedOffer;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.RejectReason;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.models.UnmoderatedOffer;
import ru.yandex.bannerstorage.harvester.queues.rtbintegration.postmoderation.services.AssemblyService;
import ru.yandex.bannerstorage.harvester.utils.JsonUtils;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonMap;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;

/**
 * @author egorovmv
 */
public final class JdbcAssemblyService implements AssemblyService {
    private static final String GET_UNSYNC_MODERATED_OFFERS_CALL = "{call dbo.sp_rtb_host_get_unsynced_assembly_objects(?)}";
    private static final String FETCH_COUNT_PARAM_NAME = "fetchCount";
    private static final String MODERATED_OFFERS_RESULT_NAME = "moderatedOffers";
    private static final String REJECT_REASONS_RESULT_NAME = "rejectReasons";

    private final NamedParameterJdbcTemplate jdbcTemplate;

    private final IntegerToIntegerMapExtractor integerToIntegerMapExtractor;
    private final ObjectWriter offerDcParamsJsonWriter;
    private final CallableStatementCreatorFactory getModeratedOffersCallFactory;
    private final List<SqlParameter> getModeratedOffersResultParams;

    public JdbcAssemblyService(@NotNull NamedParameterJdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = Objects.requireNonNull(jdbcTemplate, "jdbcTemplate");

        this.integerToIntegerMapExtractor = new IntegerToIntegerMapExtractor();

        this.offerDcParamsJsonWriter = new ObjectMapper().writerFor(
                new TypeReference<Map<String, Object>>() {
                });

        this.getModeratedOffersCallFactory = new CallableStatementCreatorFactory(
                GET_UNSYNC_MODERATED_OFFERS_CALL,
                Collections.singletonList(new SqlParameter(FETCH_COUNT_PARAM_NAME, Types.INTEGER)));
        this.getModeratedOffersResultParams = Arrays.asList(
                new SqlReturnResultSet(
                        MODERATED_OFFERS_RESULT_NAME, new ModeratedOfferRowMapper()),
                new SqlReturnResultSet(
                        REJECT_REASONS_RESULT_NAME, new ReasonWithObjectIdIdRowMapper()));
    }

    @Override
    @NotNull
    public Integer getLatestQueueId() {
        return jdbcTemplate.queryForObject(
                "SELECT MAX(queue_id) FROM dbo.t_assembly",
                Collections.emptyMap(),
                Integer.class);
    }

    @NotNull
    public AssemblyCreativeMaps getAssemblyCreativeMaps(
            @NotNull List<Integer> creativeIds, @NotNull List<Integer> creativeVersionIds) {
        Map<Integer, Integer> versionIdToCreativeId;
        if (creativeVersionIds.isEmpty())
            versionIdToCreativeId = emptyMap();
        else {
            versionIdToCreativeId = jdbcTemplate.query(
                    "SELECT nmb, creative_nmb" +
                            " FROM dbo.t_creative_version" +
                            " WHERE nmb IN (:nmbs)",
                    singletonMap("nmbs", creativeVersionIds),
                    integerToIntegerMapExtractor);
        }

        return new AssemblyCreativeMaps(
                jdbcTemplate.query(
                        "SELECT c.nmb, cv.nmb" +
                                " FROM (SELECT nmb FROM dbo.t_creative WHERE nmb IN (:nmbs)) as c" +
                                " CROSS APPLY (" +
                                "   SELECT TOP(1) nmb" +
                                "   FROM dbo.t_creative_version" +
                                "   WHERE creative_nmb = c.nmb AND status_nmb = 4 AND is_deployed = 1" +
                                "   ORDER BY nmb DESC" +
                                " ) as cv",
                        singletonMap("nmbs", creativeIds),
                        integerToIntegerMapExtractor),
                versionIdToCreativeId);
    }

    public Integer persistAssembly(@NotNull AssemblyInfo assembly, @NotNull Integer creativeVersionId) {
        return jdbcTemplate.queryForObject(
                "MERGE INTO dbo.t_assembly AS t" +
                        " USING (" +
                        "   SELECT :queueId AS queueId," +
                        "       :regionId AS regionId," +
                        "       :creativeVersionId AS creativeVersionId," +
                        "       :lastKnownImpression AS lastKnownImpression," +
                        "       :numberOfImpressions AS numberOfImpressions," +
                        "       :enqueued AS enqueued," +
                        "       :width AS width," +
                        "       :height AS height," +
                        "       CAST(:bidReqId AS NVARCHAR(30)) AS bidReqId," +
                        "       :bidReqTime AS bidReqTime," +
                        "       :creativeHits AS creativeHits" +
                        " ) AS s" +
                        " ON t.queue_id = s.queueId" +
                        " WHEN NOT MATCHED THEN" +
                        "   INSERT (queue_id, yandex_geo_nmb, creative_version_nmb, last_known_impression, number_of_impressions, enqueued, width, height, bid_req_id, bid_req_time, creative_hits)" +
                        "   VALUES (s.queueId, s.regionId, s.creativeVersionId, s.lastKnownImpression, s.numberOfImpressions, s.enqueued, s.width, s.height, s.bidReqId, s.bidReqTime, s.creativeHits);" +
                        " SELECT CAST(SCOPE_IDENTITY() AS INT)",
                new BeanPropertySqlParameterSource(assembly) {
                    @Override
                    public Object getValue(String paramName) throws IllegalArgumentException {
                        if ("creativeVersionId".equals(paramName))
                            return creativeVersionId;
                        else
                            return super.getValue(paramName);
                    }
                },
                Integer.class
        );
    }

    public void persistOffer(
            @NotNull Integer assemblyId,
            @NotNull UnmoderatedOffer offer,
            @NotNull Map<String, Object> offerDcParams,
            int orderNo) {
        jdbcTemplate.update(
                "DECLARE @rtb_hash_nmb INT;" +
                        " MERGE INTO dbo.t_rtb_hash AS t" +
                        " USING (" +
                        "   SELECT :hashId AS hashId," +
                        "       :objectType AS objectType," +
                        "       :checked AS checked," +
                        "       :moderationRequired AS moderationRequired" +
                        " ) AS s" +
                        " ON t.rtb_hash = s.hashId" +
                        " WHEN MATCHED THEN UPDATE SET @rtb_hash_nmb = t.nmb" +
                        " WHEN NOT MATCHED THEN" +
                        "   INSERT (rtb_hash, obj_type, is_checked, is_enabled, require_moderation, require_sync_with_rtb)" +
                        "   VALUES (s.hashId, s.objectType, s.checked, 1, s.moderationRequired, 0);" +
                        " IF (@rtb_hash_nmb IS NULL)" +
                        "   SET @rtb_hash_nmb = (SELECT CAST(SCOPE_IDENTITY() AS INT));" +
                        " INSERT INTO dbo.t_assembly_obj(assembly_nmb, rtb_hash_nmb, dsp_obj_id, key_name, data, hits, obj_order_nmb)" +
                        " VALUES (:assemblyId, @rtb_hash_nmb, :dspObjectId, :keyName, :data, :hits, :orderNo)",
                new MapSqlParameterSource("hashId", offer.getHashId())
                        .addValue("objectType", offer.getObjectType())
                        .addValue("checked", offer.isChecked())
                        .addValue("moderationRequired", offer.isModerationRequired())
                        .addValue("assemblyId", assemblyId)
                        .addValue("dspObjectId", offer.getDspObjectId())
                        .addValue("keyName", offer.getKeyName())
                        .addValue("data", JsonUtils.serialize(offerDcParamsJsonWriter, offerDcParams))
                        .addValue("hits", offer.getHits())
                        .addValue("orderNo", orderNo));
    }

    @Override
    @NotNull
    public List<ModeratedOffer> getUnsyncModeratedOffers(int fetchCount) {
        if (fetchCount <= 0)
            throw new IllegalArgumentException("fetchCount: " + fetchCount);

        Map<String, Object> resultSets = jdbcTemplate.getJdbcOperations()
                .call(
                        getModeratedOffersCallFactory.newCallableStatementCreator(
                                Collections.singletonMap(FETCH_COUNT_PARAM_NAME, fetchCount)),
                        getModeratedOffersResultParams);

        @SuppressWarnings("unchecked")
        List<ModeratedOffer> moderatedOffers = (List<ModeratedOffer>) resultSets.get(MODERATED_OFFERS_RESULT_NAME);
        @SuppressWarnings("unchecked")
        Map<Integer, List<RejectReason>> rejectReasons = ((List<RejectReasonWithObjectId>) resultSets.get(REJECT_REASONS_RESULT_NAME)).stream()
                .collect(
                        groupingBy(
                                RejectReasonWithObjectId::getObjectId,
                                mapping(
                                        RejectReasonWithObjectId::getRejectReason,
                                        toList())));
        moderatedOffers.forEach(
                o -> o.setReasons(rejectReasons.getOrDefault(o.getObjectId(), emptyList())));

        return moderatedOffers;
    }

    @Override
    public void markOffersAsSync(@NotNull List<Integer> objectIds) {
        jdbcTemplate.update(
                "UPDATE dbo.t_rtb_hash SET require_sync_with_rtb = 0 WHERE nmb IN (:nmbs)",
                singletonMap("nmbs", objectIds));
    }

    public static final class IntegerToIntegerMapExtractor implements ResultSetExtractor<Map<Integer, Integer>> {
        @Override
        public Map<Integer, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
            Map<Integer, Integer> result = new HashMap<>();
            while (rs.next()) {
                result.put(rs.getInt(1), rs.getInt(2));
            }
            return result;
        }
    }

    private static final class ModeratedOfferRowMapper implements RowMapper<ModeratedOffer> {
        @Override
        public ModeratedOffer mapRow(ResultSet rs, int rowNum) throws SQLException {
            ModeratedOffer result = new ModeratedOffer();
            result.setObjectId(rs.getInt("ObjectId"));
            result.setHashId(rs.getString("HashId"));
            result.setChecked(rs.getBoolean("IsChecked"));
            result.setEnabled(rs.getBoolean("IsEnabled"));
            result.setModeratorComment(rs.getString("ModeratorComment"));
            return result;
        }
    }

    private static final class RejectReasonWithObjectId {
        private final Integer objectId;
        private final RejectReason rejectReason;

        private RejectReasonWithObjectId(Integer objectId, RejectReason rejectReason) {
            this.objectId = objectId;
            this.rejectReason = rejectReason;
        }

        Integer getObjectId() {
            return objectId;
        }

        RejectReason getRejectReason() {
            return rejectReason;
        }
    }

    private static class ReasonWithObjectIdIdRowMapper implements RowMapper<RejectReasonWithObjectId> {
        private final BeanPropertyRowMapper<RejectReason> rejectReasonRowMapper;

        private ReasonWithObjectIdIdRowMapper() {
            rejectReasonRowMapper = BeanPropertyRowMapper.newInstance(RejectReason.class);
        }

        @Override
        public RejectReasonWithObjectId mapRow(ResultSet rs, int rowNum) throws SQLException {
            return new RejectReasonWithObjectId(
                    rs.getInt("ObjectId"),
                    rejectReasonRowMapper.mapRow(rs, rowNum));
        }
    }
}
