package ru.yandex.bannerstorage.harvester.queues.screenshooter;

import java.util.List;
import java.util.Objects;

import org.jetbrains.annotations.NotNull;

import ru.yandex.bannerstorage.harvester.utils.IntegerUtils;
import ru.yandex.bannerstorage.messaging.services.QueueMessageOnErrorStrategy;
import ru.yandex.bannerstorage.messaging.services.SimpleStreamQueueObserver;

/**
 * @author egorovmv
 */
public class ScreenShooterQueueObserver extends SimpleStreamQueueObserver<String> {
    private static final String QUEUE_ID = "dbo.NotifyCreativePredeployServiceQueue";
    private static final int POLL_INTERVAL_IN_MS = 10_000;
    private static final int COUNT_OF_THREADS = 20;
    private static final int BATCH_SIZE = 200;

    private final ScreenShooterSubscriber subscriber;

    public ScreenShooterQueueObserver(@NotNull ScreenShooterSubscriber subscriber) {
        super(
                QUEUE_ID,
                POLL_INTERVAL_IN_MS,
                BATCH_SIZE,
                COUNT_OF_THREADS,
                new QueueMessageOnErrorStrategy.PoisonMessageStrategy(0, false),
                String.class);
        this.subscriber = Objects.requireNonNull(subscriber, "subscriber");
    }

    @Override
    public void doProcessMessage(@NotNull String creativeVersionIdsStr) {
        List<Integer> creativeVersionIds = IntegerUtils.splitByChar(creativeVersionIdsStr, ',');
        for (Integer creativeVersionId : creativeVersionIds) {
            int attemptsLeft = 5;
            while (attemptsLeft > 0) {
                try {
                    subscriber.notify(creativeVersionId);
                    break;
                } catch (RuntimeException e) {
                    attemptsLeft--;
                    if (attemptsLeft > 0) {
                        getLogger().error(String.format("Exception while processing message, %d attempts left", attemptsLeft), e);
                        getLogger().info("Waiting 30 sec before next try..");
                        try {
                            Thread.sleep(5_000);
                        } catch (InterruptedException interruptedException) {
                            Thread.currentThread().interrupt();
                            throw new RuntimeException("Thread was interrupted", e);
                        }
                    } else {
                        getLogger().error("Exception while processing message, will not be retried", e);
                        throw e;
                    }
                }
            }
        }
    }
}
