package ru.yandex.bannerstorage.harvester.queues.screenshooter;

import java.net.URI;
import java.util.Map;
import java.util.Objects;

import javax.inject.Inject;
import javax.validation.constraints.NotNull;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.bannerstorage.harvester.integration.direct.DirectClient;
import ru.yandex.bannerstorage.harvester.queues.screenshooter.exceptions.RotorErrorException;
import ru.yandex.bannerstorage.harvester.utils.UriHelper;
import ru.yandex.bannerstorage.messaging.services.exceptions.AbortMessageProcessingException;
import ru.yandex.direct.screenshooter.client.model.ScreenShooterScreenshot;
import ru.yandex.direct.screenshooter.client.model.ScreenShooterSizeInfo;
import ru.yandex.direct.screenshooter.client.model.ScreenShooterSizesInfo;
import ru.yandex.direct.screenshooter.client.service.ScreenShooterClient;

import static java.util.Collections.singletonList;

/**
 * @author skirsanov
 */
public class ScreenShooterSubscriber {
    private static final Logger logger = LoggerFactory.getLogger(ScreenShooterSubscriber.class);

    private static final String PREVIEW_URL_QUERY = "SELECT value from s_config where name='preview_url'";

    private static final Pair<Integer, Integer> DEFAULT_ADAPTIVE_SIZE = Pair.of(600, 600);

    // В каких размерах отображать адаптивные размеры (таблицу составлял Дима Уляшев)
    private static final Map<Integer, Pair<Integer, Integer>> ADAPTIVE_SIZES =
            ImmutableMap.<Integer, Pair<Integer, Integer>>builder()
                    .put(44, Pair.of(240, 400))  // Смарт-ТГО
                    .put(45, Pair.of(336, 370))  // Смарт-плитка 1х1
                    .put(46, Pair.of(300, 300))  // Смарт-плитка 1x2
                    .put(47, Pair.of(160, 600))  // Смарт-плитка 1x3
                    .put(48, Pair.of(600, 340))  // Смарт-плитка 2x1
                    .put(49, Pair.of(600, 340))  // Смарт-плитка 2x2
                    .put(50, Pair.of(240, 600))  // Смарт-плитка 2x3
                    .put(51, Pair.of(636, 218))  // Смарт-плитка 3x1
                    .put(52, Pair.of(970, 250))  // Смарт-плитка 3x2
                    .put(53, Pair.of(970, 250))  // Смарт-плитка 3x3
                    .put(54, Pair.of(240, 400))  // Мозаика Вертикальный
                    .put(55, Pair.of(300, 600))  // Мозаика Вертикальный Большой
                    .put(56, Pair.of(160, 600))  // Мозаика Вертикальный Удлиненный
                    .put(57, Pair.of(240, 240))  // Мозаика Квадратный
                    .put(58, Pair.of(320, 280))  // Мозаика Квадратный Большой
                    .put(59, Pair.of(400, 220))  // Мозаика Горизонтальный
                    .put(60, Pair.of(600, 180))  // Мозаика Горизонтальный Большой
                    .put(61, Pair.of(900, 110))  // Мозаика Горизонтальный Удлиненный
                    .build();

    private final JdbcTemplate jdbcTemplate;
    private final ScreenShooterClient screenShooterClient;
    private final URI previewServiceUrl;
    private final DirectClient directClient;

    @Inject
    public ScreenShooterSubscriber(JdbcTemplate jdbcTemplate,
                                   ScreenShooterClient screenShooterClient,
                                   DirectClient directClient) {
        this.jdbcTemplate = jdbcTemplate;
        this.screenShooterClient = screenShooterClient;
        this.directClient = directClient;
        this.previewServiceUrl = UriHelper.fromString(
                jdbcTemplate.queryForObject(PREVIEW_URL_QUERY, String.class));
        logger.info("previewServiceUrl is {}", previewServiceUrl);
    }

    /**
     * Сделать скриншот
     *
     * @param previewInfo Метаданные скриншота
     */
    private ScreenShooterScreenshot getScreenshot(@NotNull CreativePreviewInfo previewInfo) {
        try {
            return screenShooterClient.getScreenshotFromUrl(
                    previewInfo.getPreviewUrl().toASCIIString(), previewInfo.getWidth(), previewInfo.getHeight());
        } catch (ResourceAccessException e) {
            // Если сломались из-за того что произошли какие-то сетевые проблемы, то надо попробовать обработать
            // текущее сообщение еще раз через некоторый промежуток времени (вдруг все починится). Поэтому
            // перепланируем обработку текущего сообщения
            throw new AbortMessageProcessingException(e);
        }
    }

    /**
     * Сделать скриншот данной версии креатива
     *
     * @param creativeVersionNmb Версия креатива (Integer используется для того чтобы избежать лишнего boxing-га,
     *                           так как из-вне версия креатива уже приходит как Integer)
     */
    public void notify(@NotNull Integer creativeVersionNmb) {
        CreativePreviewInfo previewInfo = getCreativePreviewInfo(creativeVersionNmb);
        // См. BANNERSTORAGE-2873
        boolean useOptimize = false;
        if (previewInfo.getWidth() == 0 || previewInfo.getHeight() == 0) {
            if (previewInfo.isSmart()) {
                // Для известных макетов мы знаем, в каких размерах имеет смысл их обскриншотивать
                // Для неизвестных будем пользоваться размером по умолчанию
                Pair<Integer, Integer> size;
                Integer layoutNmb = previewInfo.getLayoutNmb();
                if (layoutNmb != null && ADAPTIVE_SIZES.containsKey(layoutNmb)) {
                    size = ADAPTIVE_SIZES.get(layoutNmb);
                } else {
                    size = DEFAULT_ADAPTIVE_SIZE;
                }
                // Для этих размеров берём optimize, т.к. поле screenshot часто оказывается меньшего размера
                useOptimize = true;
                previewInfo.setWidth(size.getLeft());
                previewInfo.setHeight(size.getRight());
            } else {
                logger.info("Skip creativeVersion={} because width or height is 0", creativeVersionNmb);
                return;
            }
        }
        // См. BANNERSTORAGE-2881
        if (!previewInfo.getCodeMimeType().equalsIgnoreCase("text/html")) {
            logger.info("Skip creativeVersion={} because codeMimeType is not 'text/html'", creativeVersionNmb);
            return;
        }
        ScreenShooterScreenshot response = getScreenshot(previewInfo);

        if (!response.getIsDone()) {
            throw new RotorErrorException("Failed to make screenshot for creative version " + creativeVersionNmb);
        }

        // Сохранить ссылку на скриншот
        saveScreenShotData(creativeVersionNmb, response, useOptimize);

        // Если креатив смартовый, то нужно уведомить Директ об этом
        // См. BANNERSTORAGE-5661
        if (previewInfo.isSmart()) {
            directClient.notifyCreativeChanged(singletonList(previewInfo.getCreativeNmb()));
        }
    }

    /**
     * Сохранить данные о скриншоте в базе данных
     *
     * @param creativeVersionNmb    Версия креатива (Integer используется для того чтобы избежать лишнего boxing-га,
     *                              так как из-вне версия креатива уже приходит как Integer)
     * @param screenShooterResponse Ответ от скриншутилки
     * @param useOptimize           Брать поле optimize или поле screenshot
     */
    private void saveScreenShotData(@NotNull Integer creativeVersionNmb,
                                    @NotNull ScreenShooterScreenshot screenShooterResponse,
                                    boolean useOptimize) {
        ScreenShooterSizesInfo sizes = screenShooterResponse.getSizes();
        ScreenShooterSizeInfo screenshotData = useOptimize ? sizes.getOptimize() : sizes.getScreenshot();
        String screenShootUrl = screenshotData.getUrl();
        int screenShootWidth = screenshotData.getWidth();
        int screenShootHeight = screenshotData.getHeight();

        jdbcTemplate.update(
                "update t_creative_version " +
                        "set screenshot_url= ?, screenshot_width = ?, screenshot_height = ? " +
                        "where nmb = ?",
                screenShootUrl, screenShootWidth, screenShootHeight, creativeVersionNmb);
    }

    /**
     * Вернуть метаданные скриншота
     *
     * @param creativeVersionNmb Версия креатива (Integer используется для того чтобы избежать лишнего boxing-га,
     *                           так как из-вне версия креатива уже приходит как Integer)
     */
    private CreativePreviewInfo getCreativePreviewInfo(@NotNull Integer creativeVersionNmb) {
        return jdbcTemplate.queryForObject(
                "declare @creative_version_nmb int = ?;" +
                        "select " +
                        "creative_nmb," +
                        "coalesce(" +
                        "dbo.clr_uf_creative_version_get_macro_value(@creative_version_nmb, 'INITIAL_WIDTH', 1)," +
                        "dbo.uf_creative_version_get_width(@creative_version_nmb)) as width," +
                        "coalesce(" +
                        "dbo.clr_uf_creative_version_get_macro_value(@creative_version_nmb, 'INITIAL_HEIGHT', 1)," +
                        "dbo.uf_creative_version_get_height(@creative_version_nmb)) as height,\n" +
                        "coalesce(wmt.name, mt.name) as MimeType,\n" +
                        "case when tv.business_type_nmb is not null then 1 else 0 end as is_smart,\n" +
                        "tvlc.layout_nmb as layout_nmb\n" +
                        "from dbo.t_creative_version as crv with(nolock)\n" +
                        "join dbo.t_template_version as tv with(nolock) on tv.nmb = crv.template_version_nmb\n" +
                        "join dbo.t_code_version as cv with(nolock) on cv.nmb = coalesce(tv.preview_code_version_nmb," +
                        " crv.code_version_nmb)\n" +
                        "join dbo.t_code as c with(nolock) on c.nmb = cv.code_nmb\n" +
                        "join dbo.c_code_type as ct with(nolock) on ct.nmb = c.code_type_nmb\n" +
                        "join dbo.c_mime_type as mt with(nolock) on mt.nmb = ct.mime_type_nmb\n" +
                        "left join dbo.t_code_version as wcv with(nolock) on wcv.nmb = ct.wrapper_code_version_nmb\n" +
                        "left join dbo.t_code as wc with(nolock) on wc.nmb = wcv.code_nmb\n" +
                        "left join dbo.c_code_type as wct with(nolock) on wct.nmb = wc.code_type_nmb\n" +
                        "left join dbo.c_mime_type as wmt with(nolock) on wmt.nmb = wct.mime_type_nmb\n" +
                        "left join dbo.t_template_version_layout_code as tvlc with(nolock) on tvlc.nmb = crv" +
                        ".layout_code_nmb\n" +
                        "where crv.nmb = @creative_version_nmb",
                (rs, rowNum) -> {
                    Integer layoutNmb = rs.getInt("layout_nmb");
                    if (rs.wasNull()) {
                        layoutNmb = null;
                    }
                    return new CreativePreviewInfo(
                            previewServiceUrl,
                            rs.getInt(1),
                            creativeVersionNmb,
                            rs.getInt(2),
                            rs.getInt(3),
                            rs.getString(4),
                            rs.getBoolean(5),
                            layoutNmb
                    );
                },
                creativeVersionNmb);
    }

    private static class CreativePreviewInfo {
        private final URI previewUrl;
        private int width;
        private int height;
        private final String codeMimeType;
        private final boolean smart;  // Является ли креатив смарт-баннером
        private final int creativeNmb;
        @Nullable
        private final Integer layoutNmb;

        public CreativePreviewInfo(
                @NotNull URI previewServiceUrl,
                int creativeNmb,
                int creativeVersionNmb,
                int width,
                int height,
                @NotNull String codeMimeType,
                boolean smart,
                @Nullable Integer layoutNmb) {
            Objects.requireNonNull(codeMimeType, "codeMimeType");

            this.creativeNmb = creativeNmb;
            // Format: <previewServiceUrl>/page/<creative_nmb>?token=a&version=<creative_version_nmb>
            this.previewUrl = UriComponentsBuilder.fromUri(previewServiceUrl)
                    .pathSegment("page", Integer.toString(creativeNmb))
                    .queryParam("token", "a")
                    .queryParam("version", Integer.toString(creativeVersionNmb))
                    .build()
                    .toUri();
            this.width = width;
            this.height = height;
            this.codeMimeType = codeMimeType;
            this.smart = smart;
            this.layoutNmb = layoutNmb;
        }

        public URI getPreviewUrl() {
            return previewUrl;
        }

        public int getWidth() {
            return width;
        }

        public int getHeight() {
            return height;
        }

        public void setWidth(int width) {
            this.width = width;
        }

        public void setHeight(int height) {
            this.height = height;
        }

        public String getCodeMimeType() {
            return codeMimeType;
        }

        public int getCreativeNmb() {
            return creativeNmb;
        }

        public boolean isSmart() {
            return smart;
        }

        @Nullable
        public Integer getLayoutNmb() {
            return layoutNmb;
        }
    }
}
