package ru.yandex.bannerstorage.harvester.tardis.infrastracture;

import java.util.List;
import java.util.Optional;

import javax.validation.constraints.NotNull;

import ru.yandex.bannerstorage.harvester.tardis.models.ConversionResultStatus;
import ru.yandex.bannerstorage.harvester.tardis.models.FileConversionTarget;
import ru.yandex.bannerstorage.harvester.tardis.models.FileInfo;
import ru.yandex.bannerstorage.harvester.tardis.models.NewFileInfo;
import ru.yandex.bannerstorage.harvester.tardis.models.PendingFileInfo;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.ConvertedVideo;

public interface JdbcTardisService {
    /**
     * Метод получает все новые файлы для переданного инстанса
     *
     * @param instanceNmb - номер инстанса Харвестера из таблицы t_file_converter_instance
     * @return
     */
    List<NewFileInfo> getNewFiles(@NotNull Integer instanceNmb);

    /**
     * Метод получает файлы, ожидающие конвертации для переданного инстанса
     *
     * @param instanceNmb - номер инстанса Харвестера из таблицы t_file_converter_instance
     * @return
     */
    List<PendingFileInfo> getPendingFiles(@NotNull Integer instanceNmb);

    /**
     * Метод сохраняет в базу полученый урл для соответствующего файла и апдейтит статус загрузки на статус Pending
     *
     * @param file - файл, с которым нужно произвести манипуляции
     * @param url  - идентификатор таска, по которому будет формироваться урл для проверки статуса конвертации
     */
    void setUploadedFileUrl(FileInfo file, String url);

    /**
     * Метод проставляет переданному файлу указанный статус
     *
     * @param file   - файл, которому нужно обновить статус
     * @param status - статус, который нужно проставить для файла
     */
    void updateFilesStatus(FileInfo file, ConversionResultStatus status);

    /**
     * Метод добавляет инстанс Харвестера в таблицу t_file_converter_instance
     *
     * @param name - Имя машины, на которой запущено приложение
     */
    void insertIntoTFileConverterInstance(String name);

    /**
     * Метод получает номер инстанса Харвестера по переданному имени
     *
     * @param name - Имя машины, на которой запущено приложение
     * @return номер инстанса Харвестера из таблицы t_file_converter_instance
     */
    Optional<Integer> selectInstanceNmbFromTFileConverterInstance(String name);

    /**
     * Метод проставляет текущую дату как дату последней активности на инстансе Харвестера
     *
     * @param instanceNmb номер инстанса Харвестера
     */
    void updateTFileConverterInstance(Integer instanceNmb);

    /**
     * Метод проставляет инстанс Харвестера для 10 файлов с заданным статусом, ожидающих конвертации
     *
     * @param instanceNmb - инстанс Харвестера из таблицы t_fileconverter_instance
     * @param status      - статус конвертации файла
     */
    void lockFilesForInstance(@NotNull Integer instanceNmb, ConversionResultStatus status);

    void startFileProcessingTime(@NotNull FileInfo file);

    /**
     * Метод отвязывает все файлы от заданного инстанса
     *
     * @param instanceNmb - номер инстанса, от которого нужно отвязать файлы
     */
    void releaseFilesForInstance(@NotNull Integer instanceNmb);

    /**
     * Метод проверяет все ли таргеты для переданного файла были сконвертированны
     *
     * @param file - файл, для которого осуществляется проверка
     * @return - true - если не все конвертации завершились, false - если все
     */
    boolean hasUnprocessedChanges(FileInfo file);

    /**
     * Метод получает все таргеты на конвертацию для заданного файла
     *
     * @param f - файл, для которого получаем таргеты
     * @return - список таргетов на конвертацию
     */
    List<FileConversionTarget> getFileConversionTargets(@NotNull FileInfo f);

    /**
     * Метод осуществляет процесс записи результатов конвертации в базу
     *
     * @param videos   - результаты конвертации
     * @param f        - файл, для которого были получены эти результаты
     * @param duration - длительность ролика
     */
    void processResults(List<ConvertedVideo> videos, FileInfo f, int duration);
}
