package ru.yandex.bannerstorage.harvester.tardis.infrastracture.impl;

import java.net.URI;
import java.util.Objects;
import java.util.UUID;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.bannerstorage.harvester.tardis.infrastracture.VhClient;
import ru.yandex.bannerstorage.harvester.tardis.infrastracture.VhClientErrorException;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.CreatedDirectory;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.CreatedTask;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.Directory;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.Task;
import ru.yandex.bannerstorage.harvester.utils.JsonUtils;
import ru.yandex.bannerstorage.harvester.utils.RestTemplateFactory;

/**
 * HTTP клиент к API CMS видеоплатформы.
 * https://wiki.yandex-team.ru/vh/api/add-market/
 *
 * @author freakbelka
 */
public class HttpVhClientIml implements VhClient {
    private static final Logger logger = LoggerFactory.getLogger(HttpVhClientIml.class);


    private final URI serviceUrl;
    private final ObjectWriter vhTaskWriter;
    private final ObjectWriter vhDirectoryWriter;
    private final RestTemplate restTemplate;
    private final HttpHeaders headers;
    private final ObjectReader createdVhDirectoryReader;
    private final ObjectReader createdVhTaskReader;

    public HttpVhClientIml(
            @NotNull RestTemplateFactory restTemplateFactory,

            String serviceUrl,
            int connectTimeoutInMS,
            int readTimeoutInMS)
    {
        Objects.requireNonNull(restTemplateFactory);
        Objects.requireNonNull(serviceUrl);

        if (connectTimeoutInMS <= 0) {
            throw new IllegalArgumentException("connectTimeoutInMS: " + connectTimeoutInMS);
        }
        if (readTimeoutInMS <= 0) {
            throw new IllegalArgumentException("readTimeoutInMS: " + readTimeoutInMS);
        }
        this.restTemplate = restTemplateFactory.create(connectTimeoutInMS, readTimeoutInMS);

        this.serviceUrl = URI.create(serviceUrl + "/admin-api/admin/v1.0/");

        ObjectMapper jsonMapper = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        this.vhDirectoryWriter = jsonMapper.writerFor(new TypeReference<Directory>() {
        });

        this.createdVhDirectoryReader = jsonMapper.readerFor(new TypeReference<CreatedDirectory>() {
        });

        this.vhTaskWriter = jsonMapper.writerFor(new TypeReference<Task>() {
        });

        this.createdVhTaskReader = jsonMapper.readerFor(new TypeReference<CreatedTask>() {
        });
        headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
    }

    /**
     * Метод создает директорию в S3 для дальнейшей загрузки контента
     *
     * @param directory - директория, которую нужно создать
     * @return возвращает созданную директорию
     */
    @Override
    public CreatedDirectory createDirectory(Directory directory) {
        URI targetUrl = UriComponentsBuilder.fromUri(serviceUrl)
                .pathSegment("content-group.json")
                .build()
                .toUri();
        UUID requestId = UUID.randomUUID();
        try {
            String directoryJson = JsonUtils.serialize(vhDirectoryWriter, directory);
            logger.info("Creating directory");
            logger.info(
                    "Sending request (RequestId: {}, TargetUrl: {}, Directory: {})",
                    requestId, targetUrl, directoryJson);

            ResponseEntity<String> responseEntity = restTemplate.exchange(
                    targetUrl,
                    HttpMethod.POST,
                    new HttpEntity<>(directoryJson, headers),
                    String.class);

            String response = responseEntity.getBody();

            logger.info(
                    "Got response (RequestId: {}, Response: {}, Status: {})",
                    requestId, response, responseEntity.getStatusCode());
            logger.info("Created");
            return JsonUtils.deserialize(createdVhDirectoryReader, response);
        } catch (ResourceAccessException | HttpStatusCodeException e) {
            throw new VhClientErrorException(requestId, !(e instanceof HttpClientErrorException), e);
        }
    }

    /**
     * Создает таск на конвертацию в видеохостинг
     *
     * @param vhTask - таск для видео, которое нужно отконвертировать
     * @return вовзращает созданный таск
     */
    @Override
    public CreatedTask uploadVideo(Task vhTask) {
        URI targetUrl = UriComponentsBuilder.fromUri(serviceUrl)
                .pathSegment("content-group.json")
                .build()
                .toUri();
        UUID requestId = UUID.randomUUID();

        try {
            String taskJson = JsonUtils.serialize(vhTaskWriter, vhTask);
            logger.info("Uploading video");
            logger.info(
                    "Sending request (RequestId: {}, TargetUrl: {}, Directory: {})",
                    requestId, targetUrl, taskJson);

            ResponseEntity<String> responseEntity = restTemplate.exchange(
                    targetUrl,
                    HttpMethod.POST,
                    new HttpEntity<>(taskJson, headers),
                    String.class);

            String response = responseEntity.getBody();
            logger.info(
                    "Got response (RequestId: {}, Response: {}, Status: {})",
                    requestId, response, responseEntity.getStatusCode());
            logger.info("Created");
            return JsonUtils.deserialize(createdVhTaskReader, response);


        } catch (ResourceAccessException | HttpStatusCodeException e) {
            throw new VhClientErrorException(requestId, !(e instanceof HttpClientErrorException), e);
        }
    }

    /**
     * Проверяет статус выполнения конвертации для переданого uuid
     *
     * @param uuid - идентификатор таска, для которого нужно проверить статус
     * @return вовзращает таск с данным uuid
     */
    @Override
    public CreatedTask checkStatus(String uuid) {
        URI targetUrl = UriComponentsBuilder.fromUri(serviceUrl)
                .pathSegment("content-group")
                .pathSegment(uuid + ".json")
                .build()
                .toUri();
        UUID requestId = UUID.randomUUID();
        try {
            logger.info(
                    "Checking status. Sending request (RequestId: {}, TargetUrl: {})",
                    requestId, targetUrl);

            ResponseEntity<String> responseEntity = restTemplate.exchange(
                    targetUrl,
                    HttpMethod.GET,
                    null,
                    String.class);

            String response = responseEntity.getBody();

            logger.info("Got response (RequestId: {}, Response: {})", requestId, response);

            return JsonUtils.deserialize(createdVhTaskReader, response);
        } catch (ResourceAccessException | HttpStatusCodeException e) {
            throw new VhClientErrorException(requestId, !(e instanceof HttpClientErrorException), e);
        }
    }
}
