package ru.yandex.bannerstorage.harvester.tardis.infrastracture.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TimeZone;

import javax.sql.DataSource;
import javax.validation.constraints.NotNull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;

import ru.yandex.bannerstorage.harvester.tardis.infrastracture.JdbcTardisService;
import ru.yandex.bannerstorage.harvester.tardis.infrastracture.TardisServiceErrorException;
import ru.yandex.bannerstorage.harvester.tardis.models.ConversionResultStatus;
import ru.yandex.bannerstorage.harvester.tardis.models.FileConversionTarget;
import ru.yandex.bannerstorage.harvester.tardis.models.FileInfo;
import ru.yandex.bannerstorage.harvester.tardis.models.NewFileInfo;
import ru.yandex.bannerstorage.harvester.tardis.models.PendingFileInfo;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.ConvertedVideo;

import static java.util.stream.Collectors.toList;
import static ru.yandex.bannerstorage.harvester.tardis.models.ConversionResultStatus.PENDING;

/**
 * JDBC клиент для общения с базой данных
 */
public class JdbcTardisServiceImpl implements JdbcTardisService {
    public static final String CDN_NAME = "CMS";
    private final JdbcTemplate jdbcTemplate;
    private final static Logger logger = LoggerFactory.getLogger(JdbcTardisServiceImpl.class);
    private final DataSource dataSource;

    @Autowired
    public JdbcTardisServiceImpl(@NotNull DataSource dataSource) {
        this.dataSource = Objects.requireNonNull(dataSource, "dataSource");
        this.jdbcTemplate = new JdbcTemplate(dataSource);
    }

    /**
     * Метод проставляет инстанс Харвестера для 10 файлов с заданным статусом, ожидающих конвертации
     *
     * @param instanceNmb - инстанс Харвестера из таблицы t_fileconverter_instance
     * @param status      - статус конвертации файла
     */
    @Override
    public void lockFilesForInstance(@NotNull Integer instanceNmb, ConversionResultStatus status) {
        jdbcTemplate
                .update("update top (10) fc set file_converter_instance_nmb = ? from t_file_cdn fc join t_cdn c on fc.cdn_nmb = c.nmb"
                                + " where c.name = ? and fc.has_unprocessed_changes = 1 and fc.status_nmb = ? and fc.file_converter_instance_nmb is null",
                        instanceNmb, CDN_NAME, status.getValue());
    }

    /**
     * Метод проставляет время отправки файла на конвертацию
     *
     * @param file - файл, с которым нужно произвести манипуляции
     */
    @Override
    public void startFileProcessingTime(@NotNull FileInfo file) {
        jdbcTemplate
                .update("declare @time datetime = GETDATE() "
                                + " update t_file_cdn set processing_started = @time, last_action_occured = @time where nmb = ? and processing_started is null",
                        file.getFileCdnNmb());
    }

    /**
     * Метод отвязывает все файлы от заданного инстанса
     *
     * @param instanceNmb - номер инстанса, от которого нужно отвязать файлы
     */
    @Override
    public void releaseFilesForInstance(@NotNull Integer instanceNmb) {
        jdbcTemplate
                .update("update t_file_cdn set file_converter_instance_nmb = NULL where file_converter_instance_nmb = ?",
                        instanceNmb);
    }

    /**
     * Метод получает все новые файлы для переданного инстанса
     *
     * @param instanceNmb - номер инстанса Харвестера из таблицы t_file_converter_instance
     * @return
     */
    @Override
    public List<NewFileInfo> getNewFiles(@NotNull Integer instanceNmb) {
        logger.info("Retrieving new files");
        return jdbcTemplate.queryForList(
                "select top 10 fc.nmb, fc.file_nmb, fc.cdn_nmb, c.name AS cdn_name, c.upload_url, f.stillage_file_url, f.width, f.height "
                        + "from dbo.t_file f with(nolock) "
                        + "inner join t_file_cdn fc on f.nmb = fc.file_nmb "
                        + "inner join t_cdn c on fc.cdn_nmb = c.nmb "
                        + "where c.name = ? and fc.has_unprocessed_changes = 1  and fc.status_nmb = ? and fc.file_converter_instance_nmb = ? "
                        + "order by fc.file_nmb, fc.cdn_nmb",
                CDN_NAME, ConversionResultStatus.NEW.getValue(), instanceNmb).stream().map(this::convertDataToNewFile)
                .collect(toList());
    }

    /**
     * Метод получает файлы, ожидающие конвертации для переданного инстанса
     *
     * @param instanceNmb - номер инстанса Харвестера из таблицы t_file_converter_instance
     * @return
     */
    @Override
    public List<PendingFileInfo> getPendingFiles(@NotNull Integer instanceNmb) {
        logger.info("Retrieving files waiting for conversion");
        // 1. processing_started имеет тип datetime, не содержащий таймзону
        // 2. если явно не указать тайзону, timestamp получит системную таймзону из локали ОС
        // 3. и если системная таймзона не равна московской то timestamp будет показывать некорректное время
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeZone(TimeZone.getTimeZone(ZoneId.of("Europe/Moscow")));
        return jdbcTemplate.query(
                "select top 20 fc.nmb, fc.file_nmb, fc.cdn_nmb, c.name as cdn_name, c.upload_url, fc.url, f.width, f.height, fc.processing_started "
                        + "from dbo.t_file f with(nolock) "
                        + "inner join t_file_cdn fc on f.nmb = fc.file_nmb "
                        + "inner join t_cdn c on fc.cdn_nmb = c.nmb "
                        + "where c.name = ? and fc.status_nmb = ? and fc.file_converter_instance_nmb = ? "
                        + "order by fc.file_nmb, fc.cdn_nmb",
                (rs, rowNum) -> new PendingFileInfo(
                        rs.getInt("nmb"),
                        rs.getInt("file_nmb"),
                        rs.getInt("cdn_nmb"),
                        rs.getString("cdn_name"),
                        rs.getString("upload_url"),
                        rs.getString("url"),
                        rs.getInt("width"),
                        rs.getInt("height"),
                        rs.getTimestamp("processing_started", calendar)),
                CDN_NAME, ConversionResultStatus.PENDING.getValue(), instanceNmb);

    }

    private NewFileInfo convertDataToNewFile(Map<String, Object> filesMap) {
        return new NewFileInfo((int) filesMap.get("nmb"),
                (int) filesMap.get("file_nmb"),
                (int) filesMap.get("cdn_nmb"),
                (String) filesMap.get("cdn_name"),
                (String) filesMap.get("upload_url"),
                (String) filesMap.get("stillage_file_url"),
                (int) filesMap.get("width"),
                (int) filesMap.get("height"));
    }


    /**
     * Метод сохраняет в базу полученый урл для соответствующего файла и апдейтит статус загрузки на статус Pending
     *
     * @param file - файл, с которым нужно произвести манипуляции
     * @param url  - идентификатор таска, по которому будет формироваться урл для проверки статуса конвертации
     */
    @Override
    public void setUploadedFileUrl(FileInfo file, String url) {
        logger.debug("Change file status to 'pending', set url for checking status");
        jdbcTemplate.update("update t_file_cdn set status_nmb = ?, url = ? where file_nmb = ? and cdn_nmb = ?",
                PENDING.getValue(), url, file.getNmb(), file.getCdn().getNmb());
        logger.debug("Url is set");
    }

    /**
     * Метод проставляет переданному файлу указанный статус
     *
     * @param file   - файл, которому нужно обновить статус
     * @param status - статус, который нужно проставить для файла
     */
    @Override
    public void updateFilesStatus(FileInfo file, ConversionResultStatus status) {
        logger.info("Updating status for file {} and cdn {}. Setting status {}", file.getNmb(), file.getCdn().getNmb(), status.getValue());
        jdbcTemplate
                .update("update t_file_cdn set status_nmb = ? where file_nmb = ? and cdn_nmb = ?", status.getValue(),
                        file.getNmb(), file.getCdn().getNmb());
    }

    private void createResultTable(Connection con) {
        try (Statement stmt = con.createStatement()) {
            stmt.execute("if object_id('tempdb..#results') is not null "
                    + "drop table #results");
            stmt.execute("create table #results (file_nmb int not null, "
                    + "cdn_nmb int not null, "
                    + "conversion_target_nmb int not null, "
                    + "creative_version_nmb int null, "
                    + "template_parameter_nmb int null, "
                    + "url nvarchar(4000) not null, "
                    + "order_nmb tinyint not null, "
                    + "was_deleted bit not null, "
                    + "mime_type_nmb int, "
                    + "width int, "
                    + "height int, "
                    + "bitrate int, "
                    + "duration int, "
                    + "converted_file_nmb int, "
                    + "file_cdn_nmb int )");
        } catch (SQLException e) {
            logger.error("Failed to create #results table", e);
        }
    }

    private void createTmpResultTable(Connection con, String tableName) throws SQLException {
        // Создаем временную таблицу, чтобы сохранить туда результат конвертации
        try (Statement statement = con.createStatement()) {
            statement.execute("if object_id('tempdb.." + tableName + "') is not null "
                    + "drop table " + tableName);
        }
        try (Statement statement = con.createStatement()) {
            statement.execute("create table " + tableName
                    + " (id nvarchar(4000) not null PRIMARY KEY, "
                    + "url nvarchar(4000) null, "
                    + "extension nvarchar(4000) null, "
                    + "width int, "
                    + "height int, "
                    + "bitrate int, "
                    + "duration int )");
        }
    }

    private void insertIntoResutTable(Connection con, FileInfo file, String tmpTableName) throws SQLException {
        createResultTable(con);
        try (PreparedStatement stmt = con.prepareStatement("insert #results (file_nmb, cdn_nmb, conversion_target_nmb, "
                + "order_nmb, url, was_deleted, mime_type_nmb, width, height, bitrate, duration) "
                + "select " + file.getNmb() + ", " + file.getCdn().getNmb()
                + ", ct.nmb, 1, url, 0, m.mime_type_nmb, v.width, v.height, v.bitrate, v.duration "
                + "from " + tmpTableName + " v "
                + "join dbo.t_conversion_target ct on v.id like '%' + ct.name "
                + "join dbo.c_mime_type mt on mt.name = v.extension "
                + "join dbo.c_mime_type_extension m on mt.nmb = m.mime_type_nmb "))
        {
            stmt.setQueryTimeout(60);
            stmt.execute();
        }
    }

    private void insertIntoTmpResultTable(Connection con, String tableName, ConvertedVideo video, int duration)
            throws SQLException
    {
        try (PreparedStatement stmt = con.prepareStatement("insert into " + tableName
                + " (id, url, extension, width, height, bitrate, duration) values (?, ?, ?, ?, ?, ?, ?)"))
        {
            stmt.setQueryTimeout(60);
            stmt.setString(1, video.getId());
            stmt.setString(2, video.getUrl());
            stmt.setString(3, video.getType());
            stmt.setInt(4, video.getWidth());
            stmt.setInt(5, video.getHeight());
            stmt.setInt(6, video.getBitrate());
            stmt.setInt(7, duration);
            stmt.executeUpdate();
        }
    }

    /**
     * Метод проверяет все ли таргеты для переданного файла были сконвертированны
     *
     * @param file - файл, для которого осуществляется проверка
     * @return - true - если не все конвертации завершились, false - если все
     */
    @Override
    public boolean hasUnprocessedChanges(FileInfo file) {
        try {
            return jdbcTemplate
                    .queryForObject(
                            "select has_unprocessed_changes from t_file_cdn where nmb = " + file.getFileCdnNmb(),
                            Boolean.class);
        } catch (EmptyResultDataAccessException e) {
            logger.debug(String.valueOf(e));
            throw new TardisServiceErrorException(
                    "There is no record in t_file_cdn for fileCdnNmb " + file.getFileCdnNmb(), e);
        }
    }

    /**
     * Метод добавляет инстанс Харвестера в таблицу t_file_converter_instance
     *
     * @param name - Имя машины, на которой запущено приложение
     */
    @Override
    public void insertIntoTFileConverterInstance(@NotNull String name) {
        try {
            jdbcTemplate.update("declare @date datetime = GETDATE(); "
                            + "insert dbo.t_file_converter_instance (name, added, last_registered) values (?, @date, @date)",
                    name);
        } catch (Exception e) {
            logger.error("Failed to insert new instance for {}", name);
            throw new TardisServiceErrorException(e);
        }
    }

    /**
     * Метод получает номер инстанса Харвестера по переданному имени
     *
     * @param name - Имя машины, на которой запущено приложение
     * @return номер инстанса Харвестера из таблицы t_file_converter_instance
     */
    @Override
    public Optional<Integer> selectInstanceNmbFromTFileConverterInstance(@NotNull String name) {
        try {
            return Optional.of(jdbcTemplate
                    .queryForObject("select top 1 nmb from t_file_converter_instance where name = '" + name + "'",
                            Integer.class));
        } catch (EmptyResultDataAccessException e) {
            logger.debug(String.valueOf(e));
            return Optional.empty();
        }
    }

    /**
     * Метод проставляет текущую дату как дату последней активности на инстансе Харвестера
     *
     * @param instanceNmb номер инстанса Харвестера
     */
    @Override
    public void updateTFileConverterInstance(@NotNull Integer instanceNmb) {
        jdbcTemplate
                .update("update dbo.t_file_converter_instance set is_active = 1, last_registered = GETDATE() where nmb = ?",
                        instanceNmb);
    }

    /**
     * Метод получает все таргеты на конвертацию для заданного файла
     *
     * @param f - файл, для которого получаем таргеты
     * @return - список таргетов на конвертацию
     */
    @Override
    public List<FileConversionTarget> getFileConversionTargets(@NotNull FileInfo f) {
        return jdbcTemplate.queryForList(
                "select fct.file_cdn_nmb, ct.name, fct.conversion_target_nmb, fct.is_processed, fct.is_actual "
                        + "from t_file_conversion_target fct "
                        + "join t_conversion_target ct on ct.nmb = fct.conversion_target_nmb "
                        + "where fct.file_cdn_nmb = " + f.getFileCdnNmb()).stream()
                .map(m -> new FileConversionTarget((int) m.get("file_cdn_nmb"),
                        (String) m.get("name"),
                        (int) m.get("conversion_target_nmb"),
                        (boolean) m.get("is_processed"),
                        (boolean) m.get("is_actual"))).collect(toList());
    }

    /**
     * Метод осуществляет процесс записи результатов конвертации в базу
     *
     * @param videos   - результаты конвертации
     * @param f        - файл, для которого были получены эти результаты
     * @param duration - длительность ролика
     */
    @Override
    public void processResults(List<ConvertedVideo> videos, FileInfo f, int duration) {
        try (Connection con = dataSource.getConnection()) {
            // Создаем временную таблицу, чтобы сохранить туда результат конвертации
            String tableName = "#videos";
            this.createTmpResultTable(con, tableName);
            for (ConvertedVideo video : videos) {
                this.insertIntoTmpResultTable(con, tableName, video, duration);
            }

            // Формируем из результатов конвертации и данных из базы временную
            // таблицу #results, которой будут оперировать дальнейшие вызовы хранимок
            this.insertIntoResutTable(con, f, tableName);

            // Апдейтим результаты конвертации
            try (PreparedStatement stmt = con.prepareStatement("exec dbo.sp_file_update_conversion_results")) {
                stmt.execute();
            }
        } catch (SQLException e) {
            logger.error("Failed to process results", e);
            throw new TardisServiceErrorException(e);
        }
    }
}
