package ru.yandex.bannerstorage.harvester.tardis.infrastracture.impl;

import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.bannerstorage.harvester.tardis.infrastracture.JdbcTardisService;
import ru.yandex.bannerstorage.harvester.tardis.infrastracture.TardisService;
import ru.yandex.bannerstorage.harvester.tardis.infrastracture.TardisServiceErrorException;
import ru.yandex.bannerstorage.harvester.tardis.infrastracture.VhClient;
import ru.yandex.bannerstorage.harvester.tardis.models.ConversionResultStatus;
import ru.yandex.bannerstorage.harvester.tardis.models.FileConversionTarget;
import ru.yandex.bannerstorage.harvester.tardis.models.FileInfo;
import ru.yandex.bannerstorage.harvester.tardis.models.NewFileInfo;
import ru.yandex.bannerstorage.harvester.tardis.models.PendingFileInfo;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.ContentVersions;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.ConvertedVideo;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.CreatedTask;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.Directory;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.FaasAnswer;
import ru.yandex.bannerstorage.harvester.tardis.models.vh.Task;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.bannerstorage.harvester.tardis.models.ConversionResultStatus.SUCCESS;
import static ru.yandex.bannerstorage.harvester.tardis.models.ConversionResultStatus.UNMATCHED_RESULTS;

/**
 * Предоставляет интеграцию файлов BS с Видеохостингом
 *
 * @author freakbelka
 */
public class TardisServiceImpl implements TardisService {
    private static final Logger logger = LoggerFactory.getLogger(TardisServiceImpl.class);
    // см. VH-5395
    public static final int CONTENT_TYPE_ID = 15;
    public static final String PARENT_ID = "47ef9845ffeb2bfb8d65cefaf155b30a";
    // TODO: передавать только те типы, которые указаны на шаблоне для файла
    public static final String OUTPUT_FORMATS = "hls,mp4,webm,flv";

    public static final String CONVERTED_STATUS = "converted";
    public static final String FAILED_STATUS = "failed";

    // Статусы только для content_version типа DEEP_HD
    public static final String DHD_NOT_NEEDED = "dhd-not-needed";
    public static final String DHD_FAILED = "dhd-failed";

    public static final int DEFAULT_INITIAL_START_DELAY_IN_MS = 30000;
    public static final int POLL_INTERVAL_IN_MS = 30000;
    // Значение этой константы должно соответствовать имени таргета в t_conversion_target
    public static final String HLS_MASTER_M3U8 = "HLSPlaylist";
    // Возможные типы ContentVersion в ответе от CMS
    public static final String ORIGINAL_SDR = "ORIGINAL_SDR";
    public static final String DEEP_HD = "DEEP_HD";
    // Лимит ожидания deep HD роликов в минутах
    public static final int DHD_WAITING_LIMIT_MIN = 30;
    public static final int THUMBNAIL_WAITING_LIMIT_MIN = 30;
    public static final int FULL_HD_RESOLUTION = 1080;
    private final String vhId;
    private String directoryUuid;
    private final JdbcTardisService jdbcTardisService;
    private final VhClient vhClient;
    private static Integer instanceNmb = -1;
    private final Timer taskTimer;

    public TardisServiceImpl(JdbcTardisService jdbcTardisService, VhClient vhClient, String vhId) {
        this.jdbcTardisService = jdbcTardisService;
        this.vhClient = vhClient;
        this.vhId = vhId;

        taskTimer = new Timer("tardisTimer", true);
        taskTimer.schedule(new TimerTask() {
            @Override
            public void run() {
                try {
                    start();
                } catch (Exception e) {
                    logger.error("Unhandled exception during timer launch", e);
                }
            }
        }, DEFAULT_INITIAL_START_DELAY_IN_MS, POLL_INTERVAL_IN_MS);
    }

    private void createDirectory() {
        // Директория - абстрактная для нас сущность в видеохостинге, достаточно создать единожды
        if (this.directoryUuid != null) {
            return;
        }
        this.directoryUuid = vhClient
                .createDirectory(
                        new Directory("BsVideo", CONTENT_TYPE_ID, "BsVideo", PARENT_ID, UUID.randomUUID()))
                .getUuid();
    }

    /**
     * Method registers Harvester's instance in database (even thought table called t_fileconverter_instance as legacy)
     */
    private int registerInstance() {
        String hostname;
        try {
            hostname = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            throw new TardisServiceErrorException("Could not register service in DB", e);
        }
        Optional<Integer> instanceNmb = jdbcTardisService.selectInstanceNmbFromTFileConverterInstance(hostname);
        if (!instanceNmb.isPresent()) {
            logger.info("Trying registration...");
            jdbcTardisService.insertIntoTFileConverterInstance(hostname);
            instanceNmb = jdbcTardisService.selectInstanceNmbFromTFileConverterInstance(hostname);
            logger.info("Registered service as {}. Got instance id: {}", hostname, instanceNmb);
        } else {
            jdbcTardisService.updateTFileConverterInstance(instanceNmb.get());
            logger.info("Found already registered service {} in DB with instance id: {}", hostname,
                    instanceNmb);
        }
        return instanceNmb.get();
    }

    @Override
    public void start() {
        // Регистрируем инстанс харевестера
        this.instanceNmb = registerInstance();

        // Создаем директорию в видеохостинге
        createDirectory();
        if (this.directoryUuid == null || this.directoryUuid.isEmpty()) {
            return;
        }

        processNewFiles();
        processPendingFiles();

        // Снимаем lock на инстанс с файлов
        jdbcTardisService.releaseFilesForInstance(instanceNmb);
    }

    @Override
    public void processPendingFiles() {
        jdbcTardisService.lockFilesForInstance(instanceNmb, ConversionResultStatus.PENDING);
        for (PendingFileInfo f : jdbcTardisService.getPendingFiles(instanceNmb)) {
            CreatedTask createdTask = vhClient.checkStatus(f.getUrl());
            if (f.getHeight() >= FULL_HD_RESOLUTION) {
                logger.debug("File {} already has Full HD resolution. Use conversion results without CV", f.getNmb());
                processConversions(createdTask, f, false);
            } else if (f.getProcessingStarted() == null || f.getProcessingStarted()
                    .before(Timestamp.valueOf(LocalDateTime.now().minusMinutes(
                            DHD_WAITING_LIMIT_MIN)))) {
                // если DHD не возвращается дольше, чем мы готовы ждать, то берём результат без CV
                logger.info(
                        "Waited {} minutes for Deep HD results for file {}. Can't wait anymore, using results without" +
                                " CV instead",
                        DHD_WAITING_LIMIT_MIN, f.getNmb());
                processConversions(createdTask, f, false);
            } else if (getContentVersion(createdTask, DEEP_HD) != null
                    // Версия контента для DHD проставляется не сразу и по началу её может не быть, поэтому ждём
                    // Даже если появилась версия контента, то может не быть input_streams
                    && getContentVersion(createdTask, DEEP_HD).getInputStreams().size() > 0) {
                String dhdStatus = getStatus(createdTask, DEEP_HD);

                // Если DHD не нужен или для этого файла зафейлился CV, то берём результат без CV
                if (dhdStatus.equalsIgnoreCase(DHD_NOT_NEEDED) || dhdStatus.equalsIgnoreCase(DHD_FAILED)) {
                    logger.info("Can't get Deep HD results for file {}. Use results without CV instead", f.getNmb());
                    processConversions(createdTask, f, false);

                } else if (dhdStatus.equalsIgnoreCase(CONVERTED_STATUS)
                        // Проверяем, что есть Thumbnail, либо, что прошло много времени и мы его не ждём
                        && (createdTask.getThumbnail() != null || f
                        .getProcessingStarted().before(Timestamp.valueOf(LocalDateTime.now().minusMinutes(
                                THUMBNAIL_WAITING_LIMIT_MIN))))) {
                    logger.info("Got Deep HD for file {}", f.getNmb());
                    processConversions(createdTask, f, true);
                }
            }
        }
    }

    private void processConversions(CreatedTask createdTask, PendingFileInfo f, boolean hasDeepHD) {
        switch (getStatus(createdTask, ORIGINAL_SDR)) {
            case CONVERTED_STATUS:
                updateConversionResults(f, createdTask, hasDeepHD);
                break;
            case FAILED_STATUS:
                jdbcTardisService.updateFilesStatus(f, ConversionResultStatus.FAILED);
                break;
        }
    }

    @Override
    public void processNewFiles() {
        // Лочим новые файлы
        jdbcTardisService.lockFilesForInstance(instanceNmb, ConversionResultStatus.NEW);
        List<NewFileInfo> newFiles = jdbcTardisService.getNewFiles(instanceNmb);
        for (NewFileInfo f : newFiles) {
            logger.info("Got new file {} for cdn {}", f.getFileCdnNmb(), f.getCdn().getNmb());
            CreatedTask createdTask =
                    vhClient.uploadVideo(
                            new Task("BsVideoTask", "BsVideoTask", directoryUuid, CONTENT_TYPE_ID,
                                    encodeUrl(f.getStillageFileUrl()), vhId, OUTPUT_FORMATS));
            logger.info("Task has been created with uuid {}", createdTask.getUuid());
            jdbcTardisService.startFileProcessingTime(f);
            jdbcTardisService.setUploadedFileUrl(f, createdTask.getUuid());
        }
    }

    private static String encodeUrl(String url) {
        try {
            URI.create(url);
            return url;
        } catch (Exception e) {
            // значит скорее всего url не url-encoded, пробуем еще раз
            URI uri = UriComponentsBuilder.fromUriString(url).build().toUri();
            return uri.toASCIIString();
        }
    }

    private ContentVersions getContentVersion(CreatedTask task, String type) {
        return task.getContentVersions().stream().filter(cv -> cv.getType().equalsIgnoreCase(type)).findFirst()
                .orElse(null);
    }

    private FaasAnswer getFaasAnswer(CreatedTask task, String type) {
        return getContentVersion(task, type).getInputStreams().get(0).getFaasAnswer();
    }

    private String getStatus(CreatedTask task, String type) {
        return getContentVersion(task, type).getInputStreams().get(0).getStatus();
    }

    private List<ConvertedVideo> getConvertedVideos(CreatedTask task, String type) {
        return getFaasAnswer(task, type).getConvertedVideos();
    }

    private void updateConversionResults(FileInfo f, CreatedTask createdTask, boolean hasDeepHD) {
        // В случае возникновения проблем можно поискать логи по uuid созданной задачи на конвертацию
        logger.info("Start updating conversion results for task {}. FileInfo is {}. hasDeepHD={}",
                createdTask.getUuid(), JsonUtils.toJson(f), hasDeepHD);

        Set<String> expectedConversionTargetNames =
                jdbcTardisService.getFileConversionTargets(f).stream().map(FileConversionTarget::getName)
                        .collect(toSet());
        logger.info("Expected target names for task {}: {}", createdTask.getUuid(), expectedConversionTargetNames);
        List<ConvertedVideo> cvVideos;
        if (hasDeepHD) {
            // Отбираем необходимые видео в улучшенном качестве
            cvVideos = getConvertedVideos(createdTask, DEEP_HD).stream()
                    .filter(v -> expectedConversionTargetNames.stream().anyMatch(c -> v.getId().contains(c)))
                    .filter(v -> v.getHeight() > f.getHeight())
                    .collect(toList());
        } else {
            cvVideos = Collections.emptyList();
        }
        logger.info("CvVideos for task {}: {}", createdTask.getUuid(), JsonUtils.toJson(cvVideos));

        // Собираем все результаты по данному файлу
        List<ConvertedVideo> videos = Stream.concat(
                getConvertedVideos(createdTask, ORIGINAL_SDR).stream()
                        .filter(v -> expectedConversionTargetNames.stream().anyMatch(c -> v.getId().contains(c)))
                        .filter(v -> cvVideos.stream().noneMatch(cvVideo -> cvVideo.getId().equals(v.getId()))),
                cvVideos.stream()
        ).collect(toList());

        // Добавляем в список результатов отдельный таргет с мастерплейлистом
        videos.add(
                new ConvertedVideo(getContentVersion(createdTask, ORIGINAL_SDR).getOutputStreams().get(0).getUrl(), 0,
                        0, 0, "application/x-mpegURL",
                        HLS_MASTER_M3U8));
        logger.info("Videos for task {}: {}", createdTask.getUuid(), JsonUtils.toJson(videos));

        // Выбираем из списка результатов видео максимально близкое к оригиналу и добавляем к результатам
        ConvertedVideo original = videos.stream()
                .filter(v -> v.getType().contains("mp4"))
                .filter(v -> v.getHeight() <= f.getHeight()).max(
                        Comparator.comparingInt(ConvertedVideo::getHeight)).orElse(null);
        if (original == null) {
            logger.warn("Close to original video with height <= {} for task {} was not found. Will use video with " +
                    "greater height.", f.getHeight(), createdTask.getUuid());
            original = videos.stream()
                    .filter(v -> v.getType().contains("mp4"))
                    .min(Comparator.comparingInt(ConvertedVideo::getHeight)).get();
        }
        logger.info("Close to original video for task {}: {}", createdTask.getUuid(), JsonUtils.toJson(original));
        videos.add(
                new ConvertedVideo(original.getUrl(), original.getBitrate(), original.getHeight(), original.getWidth(),
                        original.getType(), "Original"));

        // Отдельно добавляем в результаты Thumbnail
        String protocol = createdTask.getThumbnail() != null && !createdTask.getThumbnail().startsWith("http") ? "http:" : "";
        videos.add(new ConvertedVideo(protocol + createdTask.getThumbnail(), 0, 0, 0,  "image/png", "Thumbnail"));
        jdbcTardisService.processResults(videos, f, getFaasAnswer(createdTask, ORIGINAL_SDR).getDuration());

        // Проверяем, что все необходимые таргеты запроцессились
        validateProcessedChanges(f);
    }

    private void validateProcessedChanges(FileInfo f) {
        if (jdbcTardisService.hasUnprocessedChanges(f)) {
            jdbcTardisService.updateFilesStatus(f, UNMATCHED_RESULTS);
            logger.info("Not every conversion targets were processed. Conversion failed.");
        } else {
            jdbcTardisService.updateFilesStatus(f, SUCCESS);
            logger.info("Conversion succeed.");
        }
    }
}
