package ru.yandex.bannerstorage.harvester.utils;

import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.web.client.RestTemplate;

/**
 * @author egorovmv
 */
public final class RestTemplateFactory {
    private final int defaultMaxConnPerRoute;
    private final int defaultMaxConnTotal;
    private final boolean useInsecureSsl;

    public RestTemplateFactory(
            int defaultMaxConnPerRoute, int defaultMaxConnTotal, boolean useInsecureSsl) {
        if (defaultMaxConnPerRoute < 0)
            throw new IllegalArgumentException("defaultMaxConnPerRoute: " + defaultMaxConnPerRoute);
        if (defaultMaxConnTotal < 0)
            throw new IllegalArgumentException("defaultMaxConnTotal: " + defaultMaxConnTotal);
        this.defaultMaxConnPerRoute = defaultMaxConnPerRoute;
        this.defaultMaxConnTotal = defaultMaxConnTotal;
        this.useInsecureSsl = useInsecureSsl;
    }

    @Bean
    private CloseableHttpClient newHttpClient(int maxConnPerRoute, int maxConnTotal) {
        HttpClientBuilder builder = HttpClientBuilder.create()
                .useSystemProperties();

        if (useInsecureSsl) {
            builder = builder.setSSLContext(SslUtils.buildInsecureSSLContext())
                    .setSSLHostnameVerifier(SslUtils.buildInsecureHostnameVerifier());
        }

        if (maxConnPerRoute > 0)
            builder = builder.setMaxConnPerRoute(maxConnPerRoute);

        if (maxConnTotal > 0)
            builder = builder.setMaxConnTotal(maxConnTotal);

        return builder.build();
    }

    public RestTemplate create(int connectTimeoutInMs, int readTimeoutInMs) {
        HttpComponentsClientHttpRequestFactory requestFactory = new HttpComponentsClientHttpRequestFactory(
                newHttpClient(defaultMaxConnTotal, defaultMaxConnPerRoute));
        requestFactory.setConnectTimeout(connectTimeoutInMs);
        requestFactory.setReadTimeout(readTimeoutInMs);
        return new RestTemplate(requestFactory);
    }

    public RestTemplate create(int connectTimeoutInMs, int readTimeoutInMs,
                               int maxConnTotal, int maxConnPerRoute) {
        HttpComponentsClientHttpRequestFactory requestFactory = new HttpComponentsClientHttpRequestFactory(
                newHttpClient(maxConnTotal, maxConnPerRoute));
        requestFactory.setConnectTimeout(connectTimeoutInMs);
        requestFactory.setReadTimeout(readTimeoutInMs);
        return new RestTemplate(requestFactory);
    }

    public static RestTemplate newInstance(int timeoutInMs) {
        return newInstance(timeoutInMs, timeoutInMs);
    }

    public static RestTemplate newInstance(int connectTimeoutInMs, int readTimeoutInMs) {
        HttpComponentsClientHttpRequestFactory requestFactory = new HttpComponentsClientHttpRequestFactory();
        requestFactory.setConnectTimeout(connectTimeoutInMs);
        requestFactory.setReadTimeout(readTimeoutInMs);
        return new RestTemplate(requestFactory);
    }
}
