package ru.yandex.bannerstorage.messaging;

import java.util.ArrayList;
import java.util.Collection;

import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Condition;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.core.type.AnnotatedTypeMetadata;

import ru.yandex.bannerstorage.messaging.controllers.QueueController;
import ru.yandex.bannerstorage.messaging.services.MessagingDispatcherService;
import ru.yandex.bannerstorage.messaging.services.NotificationQueueWatcherFactory;
import ru.yandex.bannerstorage.messaging.services.QueueMessageProcessor;
import ru.yandex.bannerstorage.messaging.services.QueueObserver;
import ru.yandex.bannerstorage.messaging.services.QueueProcessMessageScheduler;
import ru.yandex.bannerstorage.messaging.services.QueueStateProvider;
import ru.yandex.bannerstorage.messaging.services.impl.QueueNotificationCallbackExecutor;
import ru.yandex.bannerstorage.messaging.services.impl.ServiceBrokerNotificationQueueWatcherFactory;
import ru.yandex.bannerstorage.messaging.services.impl.ServiceBrokerNotificationQueueWatcherSettings;
import ru.yandex.bannerstorage.messaging.services.impl.ServiceBrokerPoisonMessageQueueObserver;
import ru.yandex.bannerstorage.messaging.services.impl.ServiceBrokerQueueMessageProcessor;
import ru.yandex.bannerstorage.messaging.services.impl.ServiceBrokerQueueOperations;
import ru.yandex.bannerstorage.messaging.services.impl.ServiceBrokerQueueStateProvider;

/**
 * @author egorovmv
 */
@Configuration
@Import({
        MessagingEnvironmentConfig.class,
        QueueNotificationCallbackExecutorConfig.class,
        QueueProcessMessageSchedulerConfig.class
})
public class MessagingConfig {
    private static final String POISON_MESSAGE_QUEUE_ID = "bannerstorage.messaging.poisonMessageService.queueId";

    @Bean
    public QueueStateProvider queueStateProvider(
            @NotNull MessagingEnvironmentConfig environmentConfig) {
        return new ServiceBrokerQueueStateProvider(environmentConfig.getJdbcTemplate());
    }

    @Bean
    public ServiceBrokerQueueOperations serviceBrokerQueueOperations(
            @NotNull MessagingEnvironmentConfig environmentConfig,
            @Value("${bannerstorage.messaging.notificationService.name}") String notificationServiceName,
            @Value("${bannerstorage.messaging.notificationService.contract}") String notificationServiceContract,
            @Value("${bannerstorage.messaging.notificationService.messageType}") String notificationServiceMessageType,
            @Value("${bannerstorage.messaging.poisonMessageService.name:}") String poisonMessageServiceName,
            @Value("${bannerstorage.messaging.poisonMessageService.contract:}") String poisonMessageServiceContract,
            @Value("${bannerstorage.messaging.poisonMessageService.messageType:}") String poisonMessageServiceMessageType) {
        return new ServiceBrokerQueueOperations(
                environmentConfig.getJdbcTemplate(),
                notificationServiceName,
                notificationServiceContract,
                notificationServiceMessageType,
                poisonMessageServiceName,
                poisonMessageServiceContract,
                poisonMessageServiceMessageType);
    }

    @Bean
    @Conditional(ServiceBrokerPoisonMessageQueueObserverCondition.class)
    public ServiceBrokerPoisonMessageQueueObserver serviceBrokerPoisonMessageQueueObserver(
            @Value("${" + POISON_MESSAGE_QUEUE_ID + "}") String queueId,
            @Value("${bannerstorage.messaging.poisonMessageService.queuePollIntervalInMS:1800000}") int pollIntervalInMS) {
        return new ServiceBrokerPoisonMessageQueueObserver(queueId, pollIntervalInMS);
    }

    @Bean
    public QueueMessageProcessor queueMessageProcessor(
            @NotNull MessagingEnvironmentConfig environmentConfig,
            @NotNull ServiceBrokerQueueOperations queueOperations) {
        return new ServiceBrokerQueueMessageProcessor(
                environmentConfig.getTransactionManager(), queueOperations);
    }

    @Bean
    public ServiceBrokerNotificationQueueWatcherSettings notificationQueueWatcherSettings(
            @Value("${bannerstorage.messaging.notificationWatcher.queueId:}") String queueId,
            @Value("${bannerstorage.messaging.notificationWatcher.batchSize:10}") int batchSize,
            @Value("${bannerstorage.messaging.notificationWatcher.receiveTimeoutInMS:60000}") int receiveTimeoutInMS,
            @Value("${bannerstorage.messaging.notificationWatcher.sleepIntervalInMS:300000}") int sleepIntervalInMS) {
        return new ServiceBrokerNotificationQueueWatcherSettings(
                queueId,
                batchSize,
                receiveTimeoutInMS,
                sleepIntervalInMS);
    }

    @Bean
    public NotificationQueueWatcherFactory notificationQueueWatcherFactory(
            @NotNull MessagingEnvironmentConfig environmentConfig,
            @NotNull ServiceBrokerNotificationQueueWatcherSettings notificationQueueWatcherSettings,
            @NotNull ServiceBrokerQueueOperations queueOperations,
            @NotNull QueueNotificationCallbackExecutor notificationCallbackExecutor) {
        return new ServiceBrokerNotificationQueueWatcherFactory(
                notificationQueueWatcherSettings,
                environmentConfig.getTransactionManager(),
                queueOperations,
                notificationCallbackExecutor);
    }

    @Bean(initMethod = "start")
    public MessagingDispatcherService messagingDispatcherService(
            @NotNull QueueStateProvider queueStateProvider,
            @NotNull QueueMessageProcessor queueMessageProcessor,
            @NotNull QueueProcessMessageScheduler processMessageScheduler,
            @NotNull NotificationQueueWatcherFactory notificationQueueWatcherFactory,
            @NotNull Collection<QueueObserver> observers) {
        return new MessagingDispatcherService(
                queueStateProvider,
                queueMessageProcessor,
                processMessageScheduler,
                notificationQueueWatcherFactory,
                new ArrayList<>(observers));
    }

    @Bean
    public QueueController queueController(MessagingDispatcherService dispatcherService) {
        return new QueueController(dispatcherService);
    }

    private static class ServiceBrokerPoisonMessageQueueObserverCondition implements Condition {
        @Override
        public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
            // Проверяем, что название очереди задано
            String queueId = context.getEnvironment()
                    .getProperty(POISON_MESSAGE_QUEUE_ID, "");
            return queueId != null && queueId.length() > 0;
        }
    }
}
