package ru.yandex.bannerstorage.messaging.services;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bannerstorage.messaging.models.QueueMonitoringState;
import ru.yandex.bannerstorage.messaging.services.exceptions.AbortMessageProcessingException;
import ru.yandex.bannerstorage.messaging.services.exceptions.AlreadyProcessedException;
import ru.yandex.bannerstorage.messaging.services.exceptions.DispatcherNotRunningException;
import ru.yandex.bannerstorage.messaging.services.exceptions.NeedRescheduleMessageProcessingException;
import ru.yandex.bannerstorage.messaging.services.exceptions.UnknownQueueException;

/**
 * @author egorovmv
 */
public final class MessagingDispatcherService implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(MessagingDispatcherService.class);

    private final QueueStateProvider queueStateProvider;
    private final QueueMessageProcessor queueMessageProcessor;
    private final QueueProcessMessageScheduler processMessageScheduler;
    private final NotificationQueueWatcher notificationQueueWatcher;
    private final Map<String, QueueState> queues;
    private volatile DispatcherState state;

    public MessagingDispatcherService(
            @NotNull QueueStateProvider queueStateProvider,
            @NotNull QueueMessageProcessor queueMessageProcessor,
            @NotNull QueueProcessMessageScheduler processMessageScheduler,
            @NotNull NotificationQueueWatcherFactory notificationQueueWatcherFactory,
            @NotNull List<QueueObserver> observers) {
        this.queueStateProvider = Objects.requireNonNull(queueStateProvider, "queueStateProvider");
        this.queueMessageProcessor = Objects.requireNonNull(queueMessageProcessor, "queueMessageProcessor");
        this.processMessageScheduler = Objects.requireNonNull(processMessageScheduler, "processMessageScheduler");
        this.notificationQueueWatcher = Objects.requireNonNull(notificationQueueWatcherFactory, "notificationQueueWatcherFactory")
                .createWatcher(this::wakeUp);
        this.queues = Objects.requireNonNull(observers, "observers")
                .stream()
                .collect(Collectors.toMap(QueueObserver::getQueueId, QueueState::new));
        this.state = DispatcherState.CREATED;
    }

    private static long getRescheduleDelayInMS(int retryCount) {
        if (retryCount > 8)
            return -1;
        return (1 << retryCount) * 1000;
    }

    public void start() {
        LOGGER.info("Starting...");

        state = DispatcherState.STARTING;

        try {
            Collection<QueueState> queues = this.queues.values();

            queues.forEach(q -> q.getObserver().start());

            queues.forEach(q -> {
                QueueObserver observer = q.getObserver();
                // Делаем рассинхронизацию разных очередей, чтобы обработка сообщений в них не запускалась
                // одновременно, а была распределена во времени
                long initialDelayInMS = (long) (Math.random() * observer.getInitialStartDelayInMS());
                processMessageScheduler.startPolling(
                        () -> {
                            AtomicInteger requestCounter = q.getRequestCounter();

                            // Сообщения в очереди уже обрабытваются?
                            if (requestCounter.get() > 0)
                                return;

                            processQueueMessages(q);
                        },
                        initialDelayInMS,
                        observer.getPollIntervalInMS()
                );
            });

            notificationQueueWatcher.start();

            state = DispatcherState.RUNNING;

            LOGGER.info("Started");
        } catch (Exception e) {
            try {
                LOGGER.error("Can't start", e);
            } finally {
                shutdown();
            }
        }
    }

    public QueueMonitoringState getQueueMonitoringState(@NotNull String queueId) {
        QueueState queue = queues.get(queueId);
        if (queue == null)
            throw new UnknownQueueException(queueId);
        return queueStateProvider.getStateOf(queue.getObserver().getQueueId());
    }

    public List<QueueMonitoringState> getQueuesMonitoringState() {
        return queues.values()
                .stream()
                .map(q -> queueStateProvider.getStateOf(q.getObserver().getQueueId()))
                .collect(Collectors.toList());
    }

    private boolean tryProcessQueueMessages(QueueObserver observer) {
        try {
            LOGGER.info("Processing messages of queue \"{}\"...", observer.getQueueId());
            queueMessageProcessor.processMessages(observer, () -> state == DispatcherState.SHUTTING_DOWN);
            LOGGER.info("Messages of queue \"{}\" processed", observer.getQueueId());
            return true;
        } catch (AlreadyProcessedException | AbortMessageProcessingException e) {
            return true;
        } catch (NeedRescheduleMessageProcessingException e) {
            return false;
        }
    }

    private void processQueueMessages(QueueState queue) {
        AtomicInteger requestCounter = queue.getRequestCounter();
        do {
            // Сбрасываем счетчик запросов на обработку для того чтобы понять
            // появились ли новые сообщения в очереди всегда сравнивать счетчик с 1
            requestCounter.set(1);

            QueueObserver observer = queue.getObserver();
            do {
                try {
                    // Если все обработалось и не требуется повторная обработка, то сбрасываем счетчик попыток
                    if (tryProcessQueueMessages(observer))
                        queue.getRetryCounter().set(0);
                    else if (state == DispatcherState.RUNNING) {
                        // В противном случае перепланируем обработку, но со сдвигом по времени в надежде на то,
                        // что в следующий раз получится
                        long rescheduleDelayInMS = getRescheduleDelayInMS(queue.getRetryCounter().incrementAndGet());
                        if (rescheduleDelayInMS > 0) {
                            LOGGER.info("Rescheduling processing messages of queue \"{}\"...", observer.getQueueId());
                            processMessageScheduler.reschedule(
                                    () -> processQueueMessages(queue),
                                    rescheduleDelayInMS);
                            LOGGER.info("Processing messages of queue \"{}\" rescheduled", observer.getQueueId());
                        }
                    }
                } catch (Throwable e) {
                    LOGGER.error(
                            String.format(
                                    "Error while processing messages of queue \"%s\".",
                                    observer.getQueueId()),
                            e);
                }
            }
            while (requestCounter.getAndSet(1) > 1); // Появились новые сообщения в очереди?
        }
        while (requestCounter.decrementAndGet() > 0); // Возможно появились новые сообщения в очереди после последней проверки
    }

    private void wakeUp(QueueState queue) {
        AtomicInteger requestCounter = queue.getRequestCounter();
        // Если никто еще не обрабытывает сообщения в очереди, то запускаем обработку
        if (requestCounter.getAndIncrement() == 0)
            processMessageScheduler.submit(() -> processQueueMessages(queue));
    }

    public void wakeUp(@NotNull String queueId) {
        Objects.requireNonNull(queueId, "queueId");

        if (state != DispatcherState.RUNNING)
            throw new DispatcherNotRunningException();

        QueueState queue = queues.get(queueId);
        if (queue == null)
            throw new UnknownQueueException(queueId);

        wakeUp(queue);
    }

    public void wakeUpAll() {
        queues.values()
                .stream()
                .filter(q -> q.getObserver().isRunning())
                .forEach(this::wakeUp);
    }

    private void closeObservers() {
        queues.values()
                .stream()
                .map(QueueState::getObserver)
                .filter(QueueObserver::isRunning)
                .forEach(o -> {
                    try {
                        o.close();
                    } catch (Exception e) {
                        LOGGER.error("Error while closing observers", e);
                    }
                });
    }

    private void shutdown() {
        LOGGER.info("Shutting down...");

        state = DispatcherState.SHUTTING_DOWN;
        try {
            try {
                notificationQueueWatcher.close();
            } finally {
                try {
                    processMessageScheduler.close();
                } finally {
                    closeObservers();
                }
            }
        } catch (Exception e) {
            LOGGER.error("Error while shutting down", e);
        }
        state = DispatcherState.STOPPED;

        LOGGER.info("Shutting downed");
    }

    @Override
    public void close() {
        if (state != DispatcherState.RUNNING)
            return;
        shutdown();
    }

    private enum DispatcherState {
        CREATED, STARTING, RUNNING, SHUTTING_DOWN, STOPPED
    }

    private static class QueueState {
        private final QueueObserver observer;
        private final AtomicInteger requestCounter;
        private final AtomicInteger retryCounter;

        private QueueState(@NotNull QueueObserver observer) {
            this.observer = observer;
            this.requestCounter = new AtomicInteger();
            this.retryCounter = new AtomicInteger();
        }

        QueueObserver getObserver() {
            return observer;
        }

        AtomicInteger getRequestCounter() {
            return requestCounter;
        }

        AtomicInteger getRetryCounter() {
            return retryCounter;
        }
    }
}
