package ru.yandex.bannerstorage.messaging.services;

import org.jetbrains.annotations.NotNull;

import ru.yandex.bannerstorage.messaging.services.exceptions.AbortMessageProcessingException;
import ru.yandex.bannerstorage.messaging.services.exceptions.AlreadyProcessedException;

/**
 * Стратегия поведения в случае ошибок, при обработке сообщения из очереди
 *
 * @author egorovmv
 */
public interface QueueMessageOnErrorStrategy {
    QueueMessageOnErrorStrategy DEFAULT = new DefaultStrategy();
    QueueMessageOnErrorStrategy IGNORE_ERROR = new IgnoreStrategy();
    QueueMessageOnErrorStrategy INFINITE_REINSERT_IN_THE_END = new ReinsertInTheEndStrategy(0);
    QueueMessageOnErrorStrategy INFINITE_POISON_START_NEW_SESSION = new PoisonMessageStrategy(0, true);
    QueueMessageOnErrorStrategy INFINITE_POISON_USE_OLD_SESSION = new PoisonMessageStrategy(0, false);

    void processError(@NotNull QueueOperations queueOperations,
                      @NotNull QueueMessage message,
                      @NotNull Throwable originalError);

    abstract class AbstractOnErrorStrategy implements QueueMessageOnErrorStrategy {
        public abstract void doProcessError(
                @NotNull QueueOperations queueOperations,
                @NotNull QueueMessage message,
                @NotNull Throwable originalError);

        @Override
        public final void processError(
                @NotNull QueueOperations queueOperations,
                @NotNull QueueMessage message,
                @NotNull Throwable originalError) {
            // Если обработка сообщения была прервана сознательно, то пробрасываем наверх для rescheduling-га
            // так высокая вероятность, что обработка следующего сообщения также не сработает и нам нужно
            // попытаться обработать данное сообщение
            if (originalError instanceof AbortMessageProcessingException)
                throw (AbortMessageProcessingException) originalError;
            else
                doProcessError(queueOperations, message, originalError);
        }
    }

    final class DefaultStrategy extends AbstractOnErrorStrategy {
        @Override
        public void doProcessError(
                @NotNull QueueOperations queueOperations,
                @NotNull QueueMessage message,
                @NotNull Throwable originalError) {
            // Wrap-им для того чтобы ошибка не логировалась дважды
            throw new AlreadyProcessedException(originalError);
        }
    }

    final class IgnoreStrategy extends AbstractOnErrorStrategy {
        @Override
        public void doProcessError(
                @NotNull QueueOperations queueOperations,
                @NotNull QueueMessage message,
                @NotNull Throwable originalError) {
            // Сознательно ничего не делаем
        }
    }

    final class PoisonMessageStrategy extends AbstractOnErrorStrategy {
        private final int maxRetryCount;
        private final boolean isNewSessionRequired;

        public PoisonMessageStrategy(int maxRetryCount, boolean isNewSessionRequired){
            this.maxRetryCount = maxRetryCount;
            this.isNewSessionRequired = isNewSessionRequired;
        }

        @Override
        public void doProcessError(
                @NotNull QueueOperations queueOperations,
                @NotNull QueueMessage message,
                @NotNull Throwable originalError) {
            if (maxRetryCount <= 0 || message.getRetryCount() <= maxRetryCount)
                queueOperations.sendToPoisonMessageService(message, isNewSessionRequired);
        }
    }

    final class ReinsertInTheEndStrategy extends AbstractOnErrorStrategy {
        private final int maxRetryCount;

        ReinsertInTheEndStrategy(int maxRetryCount) {
            this.maxRetryCount = maxRetryCount;
        }

        @Override
        public void doProcessError(
                @NotNull QueueOperations queueOperations,
                @NotNull QueueMessage message,
                @NotNull Throwable originalError) {
            if (maxRetryCount <= 0 || message.getRetryCount() <= maxRetryCount)
                queueOperations.reinsertInTheEnd(message);
        }
    }
}
