package ru.yandex.bannerstorage.messaging.services;

import java.util.Objects;

import org.jetbrains.annotations.NotNull;

import ru.yandex.bannerstorage.messaging.utils.MessageSerializer;

/**
 * Наблюдатель очереди предназначенный для простых обработчиков ожидающих только один тип сообщения и возвращающих
 * ответное сообщение
 *
 * @author egorovmv
 */
public abstract class SimpleRequestReplyQueueObserver<Request, Reply> extends AbstractQueueObserver {
    private final Class<Request> requestMessageClass;

    protected SimpleRequestReplyQueueObserver(
            @NotNull String queueId, int pollIntervalInMS, @NotNull Class<Request> requestMessageClass) {
        super(queueId, pollIntervalInMS, 1, QueueMessageOnErrorStrategy.DEFAULT);
        this.requestMessageClass = Objects.requireNonNull(requestMessageClass, "requestMessageClass");
    }

    /**
     * Обработать заданное сообщение и вернуть ответное сообщение
     */
    public abstract ReplyQueueMessage<Reply> doProcessMessage(@NotNull Request message);

    @Override
    public final void doProcessMessage(@NotNull QueueOperations queueOperations, @NotNull QueueMessage message) {
        ReplyQueueMessage<Reply> replyQueueMessage = doProcessMessage(
                MessageSerializer.unmarshal(message.getPayload(), requestMessageClass));
        queueOperations.replyTo(
                message,
                replyQueueMessage.getMessageType(),
                MessageSerializer.marshal(replyQueueMessage.getReply()));
    }
}
