package ru.yandex.bannerstorage.messaging.services.impl;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionCallback;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bannerstorage.messaging.services.NotificationQueueWatcher;
import ru.yandex.bannerstorage.messaging.services.NotificationQueueWatcherFactory;
import ru.yandex.bannerstorage.messaging.services.QueueMessage;
import ru.yandex.bannerstorage.messaging.services.exceptions.CantCreateNotificationQueueWatcherException;
import ru.yandex.bannerstorage.messaging.services.exceptions.CantProcessNotificationMessagesException;
import ru.yandex.bannerstorage.messaging.services.exceptions.DispatcherNotRunningException;
import ru.yandex.bannerstorage.messaging.services.exceptions.UnknownQueueException;
import ru.yandex.bannerstorage.messaging.utils.MessageSerializer;

/**
 * @author egorovmv
 */
public final class ServiceBrokerNotificationQueueWatcherFactory implements NotificationQueueWatcherFactory {
    private final ServiceBrokerNotificationQueueWatcherSettings settings;
    private final TransactionTemplate transactionTemplate;
    private final ServiceBrokerQueueOperations queueOperations;
    private final QueueNotificationCallbackExecutor notificationCallbackExecutor;

    public ServiceBrokerNotificationQueueWatcherFactory(
            @NotNull ServiceBrokerNotificationQueueWatcherSettings settings,
            @NotNull PlatformTransactionManager transactionManager,
            @NotNull ServiceBrokerQueueOperations queueOperations,
            QueueNotificationCallbackExecutor notificationCallbackExecutor) {
        this.settings = Objects.requireNonNull(settings, "settings");
        this.transactionTemplate = new TransactionTemplate(
                Objects.requireNonNull(transactionManager, "transactionManager"));
        this.queueOperations = Objects.requireNonNull(queueOperations, "queueOperations");
        this.notificationCallbackExecutor = Objects.requireNonNull(
                notificationCallbackExecutor, "notificationCallbackExecutor");
    }

    @Override
    public NotificationQueueWatcher createWatcher(@NotNull Consumer<String> notificationCallback) {
        // Если название очереди не задано, то возвращаем фиктивный watcher, который по факту ничего не делает
        if (settings.getNotificationQueueId() == null || settings.getNotificationQueueId().length() == 0)
            return new FakeNotificationQueueWatcher();
        else {
            return new PollingNotificationQueueWatcher(
                    Objects.requireNonNull(notificationCallback, "notificationCallback"));
        }
    }

    @XmlRootElement(name = "EVENT_INSTANCE")
    @XmlAccessorType(XmlAccessType.FIELD)
    public static class NotificationMessage {
        @XmlElement(name = "SchemaName")
        private String schemaName;
        @XmlElement(name = "ObjectName")
        private String queueName;

        public String getQueueId() {
            return schemaName + "." + queueName;
        }

        public String getSchemaName() {
            return schemaName;
        }

        public void setSchemaName(String schemaName) {
            this.schemaName = schemaName;
        }

        public String getQueueName() {
            return queueName;
        }

        public void setQueueName(String queueName) {
            this.queueName = queueName;
        }
    }

    private class PollingNotificationQueueWatcher implements NotificationQueueWatcher, Runnable {
        private static final String NOTIFICATION_MESSAGE_TYPE = "http://schemas.microsoft.com/SQL/Notifications/EventNotification";

        private final Logger logger;
        private final ExecutorService executorService;
        private final Consumer<String> notificationCallback;

        private PollingNotificationQueueWatcher(@NotNull Consumer<String> notificationCallback) {
            if (!queueOperations.isQueueExists(settings.getNotificationQueueId())) {
                throw new CantCreateNotificationQueueWatcherException(
                        String.format("Can't find queue \"%s\"", settings.getNotificationQueueId()));
            }

            this.logger = LoggerFactory.getLogger(PollingNotificationQueueWatcher.class);
            this.executorService = Executors.newSingleThreadExecutor(
                    new ThreadFactoryBuilder().setNameFormat("bs-messaging-watcher-pool-%d").build());
            this.notificationCallback = notificationCallback;
        }

        @Override
        public void start() {
            logger.info("Starting...");
            executorService.submit(this);
            logger.info("Started");
        }

        private boolean tryWaitMayBeAllBeAllRight() {
            try {
                Thread.sleep(settings.getSleepIntervalInMS());
                return true;
            } catch (InterruptedException ignored) {
                Thread.currentThread().interrupt();
                return false;
            }
        }

        @Override
        public void run() {
            TransactionCallback<Boolean> processNotificationMessages = transactionStatus -> {
                try {
                    logger.info("Waiting for notification messages...");
                    List<QueueMessage> notificationMessages = queueOperations.receiveNotificationMessages(
                            settings.getNotificationQueueId(),
                            settings.getBatchSize(),
                            settings.getReceiveTimeoutInMS());
                    logger.info("Received notification messages ({})", notificationMessages);

                    if (!notificationMessages.isEmpty()) {
                        logger.info("Sending wake up message to queue observers...");
                        notificationMessages.stream()
                                .filter(m -> m.getMessageType().equalsIgnoreCase(NOTIFICATION_MESSAGE_TYPE))
                                .map(m -> MessageSerializer.unmarshal(m.getPayload(), NotificationMessage.class))
                                .map(NotificationMessage::getQueueId)
                                .distinct()
                                .forEach(q -> {
                                    try {
                                        notificationCallbackExecutor.execute(notificationCallback, q);
                                    } catch (UnknownQueueException e) {
                                        // Хотя такого быть не должно, но если очередь не зарегистрирована, то мы
                                        // просто игнорируем эту ошибку, иначе мы не с можем обрабатывать уведомления
                                        // от других очередей
                                        logger.error("Received notification from unknown queue", e);
                                    }
                                });
                        logger.info("Wake up message sent to queue observers");
                    }
                } catch (Exception e) {
                    // Если {@link MessagingDispatcherService} еще (уже) не запущен, то это нормальная ситуация, т.е
                    // ничего логировать не надо.
                    if (!(e instanceof DispatcherNotRunningException))
                        logger.error("Can't process notification messages", e);
                    transactionStatus.setRollbackOnly();
                    throw new CantProcessNotificationMessagesException(e);
                }

                return true;
            };

            logger.info("Begin event loop");
            try {
                while (!Thread.currentThread().isInterrupted()) {
                    try {
                        transactionTemplate.execute(processNotificationMessages);
                    } catch (Exception e) {
                        // Если мы еще не залогировали данную ошибку, то логируем ее
                        if (!(e instanceof CantProcessNotificationMessagesException))
                            logger.error("Can't execute transaction", e);

                        tryWaitMayBeAllBeAllRight();
                    }
                }
            } finally {
                logger.info("End event loop");
            }
        }

        private boolean shutdown() {
            try {
                logger.info("Shutting down...");

                executorService.shutdown();

                try {
                    executorService.awaitTermination(Integer.MAX_VALUE, TimeUnit.MILLISECONDS);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }

                // Считаем, что остановили, даже если нас прервали на ожидании
                // Вроде все логично, все равно остановится, чуть позже
                logger.info("Shutting downed");

                return true;
            } catch (Exception e) {
                logger.error("Can't shutdown", e);
                return false;
            }
        }

        @Override
        public void close() {
            shutdown();
        }
    }

    public static class FakeNotificationQueueWatcher implements NotificationQueueWatcher {
        @Override
        public void start() {
        }

        @Override
        public void close() {
        }
    }
}
