package ru.yandex.bannerstorage.messaging.services.impl;

import java.sql.SQLException;
import java.util.Map;
import java.util.Objects;
import java.util.function.BooleanSupplier;

import org.jetbrains.annotations.NotNull;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionCallback;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bannerstorage.messaging.services.QueueMessageProcessor;
import ru.yandex.bannerstorage.messaging.services.QueueObserver;
import ru.yandex.bannerstorage.messaging.services.QueueOperations;
import ru.yandex.bannerstorage.messaging.services.exceptions.AbortMessageProcessingException;
import ru.yandex.bannerstorage.messaging.services.exceptions.AlreadyProcessedException;

/**
 * @author egorovmv
 */
public final class ServiceBrokerQueueMessageProcessor implements QueueMessageProcessor {
    private final TransactionTemplate transactionTemplate;
    private final QueueOperations queueOperations;

    public ServiceBrokerQueueMessageProcessor(
            @NotNull PlatformTransactionManager transactionManager, @NotNull QueueOperations queueOperations) {
        this.transactionTemplate = new TransactionTemplate(
                Objects.requireNonNull(transactionManager, "transactionManager"));
        this.queueOperations = Objects.requireNonNull(queueOperations, "queueOperations");
    }

    @Override
    public void processMessages(@NotNull QueueObserver queueObserver, @NotNull BooleanSupplier isShutdownRequested) {
        try {
            Map<String, Object> localState = queueObserver.newLocalState();
            TransactionCallback<Boolean> processMessages = transactionStatus -> {
                try {
                    return queueObserver.processMessages(queueOperations, localState);
                } catch (Throwable e) {
                    transactionStatus.setRollbackOnly();
                    throw e;
                }
            };
            //noinspection StatementWithEmptyBody
            while (!isShutdownRequested.getAsBoolean() && transactionTemplate.execute(processMessages)) {
            }
        } catch (AlreadyProcessedException e) {
            for (Throwable current = e; current != null; current = current.getCause()) {
                // Если deadlock, то пытаемся повторно обработать сообщения через какое-то время
                if (current instanceof SQLException && ((SQLException) current).getErrorCode() == 1205)
                    throw new AbortMessageProcessingException(e);
            }
            throw e;
        }
    }
}
