package ru.yandex.bannerstorage.messaging.services.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bannerstorage.messaging.services.QueueMessage;
import ru.yandex.bannerstorage.messaging.services.QueueOperations;
import ru.yandex.bannerstorage.messaging.utils.MessageSerializer;

/**
 * @author egorovmv
 */
public final class ServiceBrokerQueueOperations implements QueueOperations {
    private static final Logger LOGGER = LoggerFactory.getLogger(ServiceBrokerQueueOperations.class);

    private final JdbcTemplate jdbcTemplate;
    private final String notificationService;
    private final String notificationContract;
    private final String notificationMessageType;
    private final String poisonMessageService;
    private final String poisonMessageContract;
    private final String poisonMessageType;
    private final QueueMessageRowMapper rowMapper;

    public ServiceBrokerQueueOperations(
            @NotNull JdbcTemplate jdbcTemplate,
            @NotNull String notificationService,
            @NotNull String notificationContract,
            @NotNull String notificationMessageType,
            @NotNull String poisonMessageService,
            @NotNull String poisonMessageContract,
            @NotNull String poisonMessageType) {
        this.jdbcTemplate = Objects.requireNonNull(jdbcTemplate, "jdbcTemplate");
        this.notificationService = Objects.requireNonNull(notificationService, "notificationService");
        this.notificationContract = Objects.requireNonNull(notificationContract, "notificationContract");
        this.notificationMessageType = Objects.requireNonNull(notificationMessageType, "notificationMessageType");
        this.poisonMessageService = Objects.requireNonNull(poisonMessageService, "poisonMessageService");
        this.poisonMessageContract = Objects.requireNonNull(poisonMessageContract, "poisonMessageContract");
        this.poisonMessageType = Objects.requireNonNull(poisonMessageType, "poisonMessageType");
        this.rowMapper = new QueueMessageRowMapper();
    }

    /**
     * Проверить существует ли очередь
     *
     * @param queueId Название очереди
     * @return true, если очередь существует
     */
    public boolean isQueueExists(@NotNull String queueId) {
        return jdbcTemplate.queryForObject(
                "SELECT CAST(\n" +
                        "CASE\n" +
                        "WHEN exists(SELECT * FROM sys.service_queues WHERE object_id = OBJECT_ID(?)) THEN 1\n" +
                        "ELSE 0\n" +
                        "END AS BIT)",
                Boolean.class,
                queueId);
    }

    public List<QueueMessage> receiveNotificationMessages(
            @NotNull String queueId, int batchSize, int timeoutInMS) {
        return jdbcTemplate.query(
                String.format(
                        "WAITFOR (RECEIVE TOP(?)\n" +
                                "CAST(conversation_handle AS VARCHAR(MAX))," +
                                "service_name," +
                                "service_contract_name," +
                                "message_type_name," +
                                "message_enqueue_time," +
                                "CAST(message_body AS NVARCHAR(MAX)) as message_body," +
                                "0 as retry_count\n" +
                                "FROM %s" +
                                "), TIMEOUT %s",
                        queueId,
                        Integer.toString(timeoutInMS)),
                rowMapper,
                batchSize);
    }

    @Override
    public List<QueueMessage> receiveMessages(
            @NotNull String queueId, int batchSize, int timeoutInMS) {
        LOGGER.info("Receiving messages from queue \"{}\"...", queueId);

        List<QueueMessage> messages = jdbcTemplate.query(
                String.join(
                        ";\n",
                        "DECLARE @messages TABLE (" +
                                "conversation_handle VARCHAR(64)," +
                                "service_name SYSNAME," +
                                "service_contract_name SYSNAME," +
                                "queuing_order BIGINT," +
                                "message_type_name SYSNAME," +
                                "message_enqueue_time DATETIME," +
                                "message_body XML)",
                        String.format(
                                "WAITFOR (RECEIVE TOP(?)\n" +
                                        "CAST(conversation_handle AS VARCHAR(MAX))," +
                                        "service_name," +
                                        "service_contract_name," +
                                        "queuing_order," +
                                        "message_type_name," +
                                        "message_enqueue_time," +
                                        "cast(message_body AS XML)\n" +
                                        "FROM %s\n" +
                                        "INTO @messages" +
                                        "), TIMEOUT %s",
                                queueId,
                                Integer.toString(timeoutInMS)),
                        "SELECT\n" +
                                "conversation_handle," +
                                "service_name," +
                                "service_contract_name," +
                                "message_type_name," +
                                "message_enqueue_time," +
                                "dbo.uf_unwrap_queue_message(message_body)," +
                                "COALESCE(message_body.value('(/message/@retry_count)[1]', 'bigint'), 0)\n" +
                                "FROM @messages\n" +
                                "ORDER BY queuing_order"),
                rowMapper,
                batchSize);

        LOGGER.info("{} messages received from queue \"{}\"", messages.size(), queueId);

        return messages;
    }

    private void doSendMessage(
            @NotNull String fromService,
            @NotNull String toService,
            @NotNull String contract,
            @NotNull String messageType,
            @NotNull String payload,
            @NotNull Integer retryCount) {
        String sql = String.format(
                "DECLARE @ch UNIQUEIDENTIFIER;\n" +
                        "BEGIN DIALOG @ch\n" +
                        "FROM SERVICE [%s]\n" +
                        "TO SERVICE ?\n" +
                        "ON CONTRACT [%s]\n" +
                        "WITH ENCRYPTION = OFF;\n" +
                        "DECLARE @message NVARCHAR(MAX) = dbo.uf_wrap_queue_message_with_retry_count(?, ?);\n" +
                        "SEND ON CONVERSATION @ch\n" +
                        "MESSAGE TYPE [%s]\n" +
                        "(@message);\n" +
                        "END CONVERSATION @ch;",
                fromService,
                contract,
                messageType);
        jdbcTemplate.update(sql, toService, payload, retryCount);
    }

    private void sendMessage(
            @NotNull String fromService,
            @NotNull String toService,
            @NotNull String contract,
            @NotNull String messageId,
            @NotNull String messageType,
            @NotNull String payload,
            @NotNull Integer retryCount) {
        Objects.requireNonNull(fromService, "fromService");
        Objects.requireNonNull(toService, "toService");
        Objects.requireNonNull(contract, "contract");
        Objects.requireNonNull(messageId, "messageId");
        Objects.requireNonNull(messageType, "messageType");
        Objects.requireNonNull(payload, "payload");
        Objects.requireNonNull(retryCount, "retryCount");

        LOGGER.info(
                "Sending message (" +
                        "MessageId: \"{}\"" +
                        ", FromService: \"{}\"" +
                        ", ToService: \"{}\"" +
                        ", Contract: \"{}\"" +
                        ", MessageType: \"{}\"" +
                        ", Payload: \"{}\"" +
                        ", RetryCount: \"{}\"" +
                        ")...",
                messageId,
                fromService,
                toService,
                contract,
                messageType,
                payload,
                retryCount);

        doSendMessage(fromService, toService, contract, messageType, payload, retryCount);

        LOGGER.info("Message sent (MessageId: \"{}\")", messageId);
    }

    @Override
    public <T> void sendMessage(@NotNull String targetService, @NotNull T payload) {
        sendMessage(
                notificationService,
                targetService,
                notificationContract,
                UUID.randomUUID().toString(),
                notificationMessageType,
                MessageSerializer.marshal(payload),
                0);
    }

    private void doReplyTo(
            @NotNull ServiceBrokerQueueMessage message, @NotNull String messageType, @NotNull String reply) {
        Objects.requireNonNull(message, "message");
        Objects.requireNonNull(messageType, "messageType");
        Objects.requireNonNull(reply, "reply");

        String replyId = UUID.randomUUID().toString();

        LOGGER.info(
                "Sending reply to message (MessageId: \"{}\", ReplyId: \"{}\", MessageType: \"{}\", Reply: \"{}\")...",
                message.getMessageId(),
                replyId,
                messageType,
                reply);

        String sql = String.format(
                "DECLARE @message NVARCHAR(MAX) = dbo.uf_wrap_queue_message(?);\n" +
                        "DECLARE @ch UNIQUEIDENTIFIER = ?;\n" +
                        "SEND ON CONVERSATION @ch\n" +
                        "MESSAGE TYPE [%s]\n" +
                        "(@message);\n" +
                        "END CONVERSATION @ch;",
                messageType);
        jdbcTemplate.update(sql, reply, message.getConversationId());

        LOGGER.info(
                "Reply sent to message (MessageId: \"{}\", ReplyId: \"{}\")",
                message.getMessageId(), replyId);
    }

    @Override
    public void replyTo(
            @NotNull QueueMessage message, @NotNull String messageType, @NotNull String reply) {
        doReplyTo(
                (ServiceBrokerQueueMessage) message, messageType, reply);
    }

    @Override
    public void sendToPoisonMessageService(@NotNull QueueMessage message, boolean isNewSessionRequired) {
        Objects.requireNonNull(message, "message");

        LOGGER.info("Sending message to poison message service (MessageId: \"{}\")...", message.getMessageId());

        doSendMessage(
                notificationService,
                poisonMessageService,
                poisonMessageContract,
                poisonMessageType,
                MessageSerializer.marshal(
                        ServiceBrokerPoisonMessage.fromQueueMessage(
                                (ServiceBrokerQueueMessage) message, isNewSessionRequired)),
                0);

        LOGGER.info("Message sent to poison message service (MessageId: \"{}\")", message.getMessageId());
    }

    private String getInitiatorConversationId(@NotNull String targetConversationId) {
        return jdbcTemplate.queryForObject(
                "SELECT conversation_handle" +
                        " FROM sys.conversation_endpoints" +
                        " WHERE conversation_id = (SELECT conversation_id FROM sys.conversation_endpoints WHERE conversation_handle = ?)" +
                        "   AND is_initiator = 1",
                String.class,
                targetConversationId);
    }

    private void doReinsertMessageInTheEnd(
            @NotNull String targetConversationId,
            @NotNull String targetService,
            @NotNull String contract,
            @NotNull String messageType,
            @NotNull String payload,
            @NotNull Integer retryCount,
            boolean isNewSessionRequired) {
        if (isNewSessionRequired) {
            doSendMessage(
                    notificationService,
                    targetService,
                    contract,
                    messageType,
                    payload,
                    retryCount + 1);
        } else {
            String initiatorConversationId = getInitiatorConversationId(targetConversationId);

            String sql = String.format(
                    "DECLARE @message XML = dbo.uf_wrap_queue_message_with_retry_count(?, ?);\n" +
                            "DECLARE @ch UNIQUEIDENTIFIER = ?;\n" +
                            "SEND ON CONVERSATION @ch\n" +
                            "MESSAGE TYPE [%s]\n" +
                            "(@message);",
                            messageType);
            jdbcTemplate.update(
                    sql,
                    payload,
                    retryCount + 1,
                    initiatorConversationId);
        }
    }

    private void doReinsertMessageInTheEnd(@NotNull ServiceBrokerQueueMessage message) {
        doReinsertMessageInTheEnd(
                message.getConversationId(),
                message.getServiceName(),
                message.getContractName(),
                message.getMessageType(),
                message.getPayload(),
                message.getRetryCount(),
                false);
    }

    private void doReinsertMessageInTheEnd(@NotNull ServiceBrokerPoisonMessage message) {
        doReinsertMessageInTheEnd(
                message.getConversationId(),
                message.getServiceName(),
                message.getContractName(),
                message.getMessageType(),
                message.getPayload(),
                message.getRetryCount(),
                message.isNewSessionRequired());
    }

    @Override
    public void reinsertInTheEnd(@NotNull QueueMessage message) {
        Objects.requireNonNull(message, "message");

        LOGGER.info("Reinserting message (MessageId: \"{}\")...", message.getMessageId());

        if (message instanceof ServiceBrokerQueueMessage)
            doReinsertMessageInTheEnd((ServiceBrokerQueueMessage) message);
        else if (message instanceof ServiceBrokerPoisonMessage)
            doReinsertMessageInTheEnd((ServiceBrokerPoisonMessage) message);
        else
            throw new AssertionError("Unknown queue message: " + message);

        LOGGER.info("Message reinserted (MessageId: \"{}\")", message.getMessageId());
    }

    @Override
    public void endSession(QueueMessage message) {
        jdbcTemplate.update(
                "DECLARE @ch UNIQUEIDENTIFIER = ?;\n" +
                        "END CONVERSATION @ch",
                ((ServiceBrokerQueueMessage) message).getConversationId());
    }

    private static class QueueMessageRowMapper implements RowMapper<QueueMessage> {
        @Override
        public QueueMessage mapRow(ResultSet resultSet, int i) throws SQLException {
            ServiceBrokerQueueMessage result = new ServiceBrokerQueueMessage(
                    resultSet.getString(1),
                    resultSet.getString(2),
                    resultSet.getString(3),
                    UUID.randomUUID().toString(),
                    resultSet.getString(4),
                    resultSet.getTimestamp(5),
                    resultSet.getString(6),
                    resultSet.getInt(7));
            LOGGER.info("Received message ({})", result);
            return result;
        }
    }
}
