package ru.yandex.bannerstorage.messaging.services.schedulers;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bannerstorage.messaging.services.QueueProcessMessageScheduler;

/**
 * @author egorovmv
 */
public final class ScheduledExecutorProcessMessageScheduler implements QueueProcessMessageScheduler {
    private static final Logger LOGGER = LoggerFactory.getLogger(ScheduledExecutorProcessMessageScheduler.class);

    private final ScheduledExecutorService executorService;

    public ScheduledExecutorProcessMessageScheduler(int waitTimeToCpuTime) {
        if (waitTimeToCpuTime < 0)
            throw new IllegalArgumentException("waitTimeToCpuTime");
        executorService = Executors.newScheduledThreadPool(
                Runtime.getRuntime().availableProcessors() * (1 + waitTimeToCpuTime),
                new ThreadFactoryBuilder().setNameFormat("bs-messaging-worker-pool-%d").build());
    }

    @Override
    public void startPolling(@NotNull Runnable processMessageCallback, long initialDelayInMS, long pollIntervalInMS) {
        executorService.scheduleWithFixedDelay(
                processMessageCallback, initialDelayInMS, pollIntervalInMS, TimeUnit.MILLISECONDS);
    }

    @Override
    public void reschedule(@NotNull Runnable processMessageCallback, long delayInMS) {
        executorService.schedule(processMessageCallback, delayInMS, TimeUnit.MILLISECONDS);
    }

    @Override
    public void submit(@NotNull Runnable processMessageCallback) {
        executorService.submit(processMessageCallback);
    }

    @Override
    public void close() {
        try {
            executorService.shutdown();
            executorService.awaitTermination(Integer.MAX_VALUE, TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            LOGGER.error("Can't shutdown", e);
        }
    }
}
